import os
import mimetypes
from django.shortcuts import render, get_object_or_404, redirect
from django.http import HttpResponse, HttpResponseRedirect
from django.views.generic import ListView, DetailView, CreateView, UpdateView, DeleteView, TemplateView, View
from django.urls import reverse_lazy, reverse
from django.contrib import messages
from django.contrib.auth.mixins import LoginRequiredMixin, UserPassesTestMixin
from django.db.models import Q, Sum, Count, F
from django.http import JsonResponse, HttpResponseRedirect
from django.views.decorators.http import require_http_methods
from django.contrib.auth.decorators import login_required
from django.utils.decorators import method_decorator
from django.views.generic.edit import FormMixin
from django.views.generic.detail import SingleObjectMixin
from django.views.generic import ListView, DetailView, CreateView, UpdateView, DeleteView, TemplateView, View
from django.utils import timezone

from .models import Supplier, SupplierCategory, SupplierContact, SupplierDocument, SupplierNote
from .forms import (
    SupplierForm, SupplierContactForm, SupplierDocumentForm, 
    SupplierNoteForm, SupplierCategoryForm
)
from branches.models import Branch

class SupplierListView(LoginRequiredMixin, ListView):
    model = Supplier
    template_name = 'suppliers/supplier_list.html'
    context_object_name = 'suppliers'
    paginate_by = 20
    
    def get_queryset(self):
        queryset = Supplier.objects.all()
        
        # Filter by status
        status = self.request.GET.get('status')
        if status == 'active':
            queryset = queryset.filter(is_active=True)
        elif status == 'inactive':
            queryset = queryset.filter(is_active=False)
        
        # Filter by category
        category_id = self.request.GET.get('category')
        if category_id:
            queryset = queryset.filter(category_id=category_id)
        
        # Filter by preferred status
        preferred = self.request.GET.get('preferred')
        if preferred == 'yes':
            queryset = queryset.filter(preferred=True)
        elif preferred == 'no':
            queryset = queryset.filter(preferred=False)
        
        # Search functionality
        search_query = self.request.GET.get('q')
        if search_query:
            queryset = queryset.filter(
                Q(name__icontains=search_query) |
                Q(company_name__icontains=search_query) |
                Q(email__icontains=search_query) |
                Q(phone__icontains=search_query) |
                Q(tax_identification_number__icontains=search_query) |
                Q(vat_number__icontains=search_query)
            )
        
        # Ordering
        order_by = self.request.GET.get('order_by', 'name')
        if order_by in [f.name for f in Supplier._meta.fields]:
            queryset = queryset.order_by(order_by)
        
        return queryset.select_related('category').prefetch_related('branches')
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['categories'] = SupplierCategory.objects.filter(is_active=True)
        context['current_status'] = self.request.GET.get('status', '')
        context['search_query'] = self.request.GET.get('q', '')
        context['preferred_filter'] = self.request.GET.get('preferred', '')
        context['category_filter'] = self.request.GET.get('category', '')
        return context

class SupplierDetailView(LoginRequiredMixin, DetailView):
    model = Supplier
    template_name = 'suppliers/supplier_detail.html'
    context_object_name = 'supplier'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        supplier = self.get_object()
        
        # Add forms for related models
        context['contact_form'] = SupplierContactForm(initial={'supplier': supplier})
        context['document_form'] = SupplierDocumentForm(initial={'supplier': supplier})
        context['note_form'] = SupplierNoteForm(initial={'supplier': supplier, 'created_by': self.request.user})
        
        # Add related data
        context['contacts'] = supplier.contacts.all()
        context['documents'] = supplier.documents.all()
        context['notes'] = supplier.supplier_notes.all().select_related('created_by')
        
        # Add purchase history (you'll need to implement this based on your purchase model)
        # context['purchase_orders'] = supplier.purchase_orders.all().order_by('-order_date')[:10]
        # context['total_purchases'] = supplier.purchase_orders.aggregate(total=Sum('total_amount'))['total'] or 0
        
        return context

class SupplierCreateView(LoginRequiredMixin, CreateView):
    model = Supplier
    form_class = SupplierForm
    template_name = 'suppliers/supplier_form.html'
    
    def form_valid(self, form):
        form.instance.created_by = self.request.user
        response = super().form_valid(form)
        messages.success(self.request, f'Supplier "{self.object.name}" was created successfully.')
        return response
    
    def get_success_url(self):
        return reverse('suppliers:supplier_detail', kwargs={'pk': self.object.pk})

class SupplierUpdateView(LoginRequiredMixin, UpdateView):
    model = Supplier
    form_class = SupplierForm
    template_name = 'suppliers/supplier_form.html'
    
    def form_valid(self, form):
        response = super().form_valid(form)
        messages.success(self.request, f'Supplier "{self.object.name}" was updated successfully.')
        return response
    
    def get_success_url(self):
        return reverse('suppliers:supplier_detail', kwargs={'pk': self.object.pk})

class SupplierDeleteView(LoginRequiredMixin, UserPassesTestMixin, DeleteView):
    model = Supplier
    template_name = 'suppliers/supplier_confirm_delete.html'
    success_url = reverse_lazy('suppliers:supplier_list')
    
    def test_func(self):
        return self.request.user.is_staff
    
    def delete(self, request, *args, **kwargs):
        supplier = self.get_object()
        messages.success(request, f'Supplier "{supplier.name}" was deleted successfully.')
        return super().delete(request, *args, **kwargs)

# Supplier Contact Views
class SupplierContactCreateView(LoginRequiredMixin, CreateView):
    model = SupplierContact
    form_class = SupplierContactForm
    template_name = 'suppliers/contact_form.html'
    
    def form_valid(self, form):
        supplier = get_object_or_404(Supplier, pk=self.kwargs['supplier_pk'])
        form.instance.supplier = supplier
        response = super().form_valid(form)
        messages.success(self.request, 'Contact was added successfully.')
        return response
    
    def get_success_url(self):
        return reverse('supplier_detail', kwargs={'pk': self.kwargs['supplier_pk']})
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['supplier'] = get_object_or_404(Supplier, pk=self.kwargs['supplier_pk'])
        return context

class SupplierContactUpdateView(LoginRequiredMixin, UpdateView):
    model = SupplierContact
    form_class = SupplierContactForm
    template_name = 'suppliers/contact_form.html'
    
    def form_valid(self, form):
        response = super().form_valid(form)
        messages.success(self.request, 'Contact was updated successfully.')
        return response
    
    def get_success_url(self):
        return reverse('supplier_detail', kwargs={'pk': self.object.supplier.pk})
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['supplier'] = self.object.supplier
        return context

class SupplierContactDeleteView(LoginRequiredMixin, UserPassesTestMixin, DeleteView):
    model = SupplierContact
    template_name = 'suppliers/contact_confirm_delete.html'
    
    def test_func(self):
        return self.request.user.is_staff
    
    def delete(self, request, *args, **kwargs):
        response = super().delete(request, *args, **kwargs)
        messages.success(request, 'Contact was deleted successfully.')
        return response
    
    def get_success_url(self):
        return reverse('supplier_detail', kwargs={'pk': self.object.supplier.pk})

# Supplier Document Views (similar pattern for other related models)
class SupplierDocumentCreateView(LoginRequiredMixin, CreateView):
    model = SupplierDocument
    form_class = SupplierDocumentForm
    template_name = 'suppliers/document_form.html'
    
    def form_valid(self, form):
        supplier = get_object_or_404(Supplier, pk=self.kwargs['supplier_pk'])
        form.instance.supplier = supplier
        form.instance.uploaded_by = self.request.user
        response = super().form_valid(form)
        messages.success(self.request, 'Document was uploaded successfully.')
        return response
    
    def get_success_url(self):
        return reverse('supplier_detail', kwargs={'pk': self.kwargs['supplier_pk']})
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['supplier'] = get_object_or_404(Supplier, pk=self.kwargs['supplier_pk'])
        return context

class SupplierDocumentUpdateView(LoginRequiredMixin, UpdateView):
    model = SupplierDocument
    form_class = SupplierDocumentForm
    template_name = 'suppliers/document_form.html'
    
    def form_valid(self, form):
        messages.success(self.request, 'Document updated successfully.')
        return super().form_valid(form)
    
    def get_success_url(self):
        return reverse('suppliers:supplier_detail', kwargs={'pk': self.object.supplier.pk})
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['supplier'] = self.object.supplier
        context['title'] = 'Update Document'
        return context

class SupplierDocumentDeleteView(LoginRequiredMixin, DeleteView):
    model = SupplierDocument
    template_name = 'suppliers/document_confirm_delete.html'
    
    def test_func(self):
        document = self.get_object()
        return self.request.user.is_staff or self.request.user == document.uploaded_by
    
    def delete(self, request, *args, **kwargs):
        document = self.get_object()
        supplier_pk = document.supplier.pk
        document.delete()
        messages.success(request, 'Document deleted successfully.')
        return HttpResponseRedirect(reverse('suppliers:supplier_detail', kwargs={'pk': supplier_pk}))
    
    def get_success_url(self):
        return reverse('suppliers:supplier_detail', kwargs={'pk': self.object.supplier.pk})

@login_required
def supplier_document_download(request, pk):
    """View to download a supplier document."""
    document = get_object_or_404(SupplierDocument, pk=pk)
    
    # Check if the user has permission to download the document
    if not (request.user.is_staff or request.user == document.uploaded_by):
        messages.error(request, 'You do not have permission to download this document.')
        return redirect('suppliers:supplier_detail', pk=document.supplier.pk)
    
    # Get the file path and open the file
    file_path = document.file.path
    if not os.path.exists(file_path):
        messages.error(request, 'The requested file does not exist.')
        return redirect('suppliers:supplier_detail', pk=document.supplier.pk)
    
    # Determine the content type based on file extension
    content_type, _ = mimetypes.guess_type(file_path)
    if not content_type:
        content_type = 'application/octet-stream'
    
    # Open the file in binary mode for reading
    with open(file_path, 'rb') as f:
        response = HttpResponse(f.read(), content_type=content_type)
        
    # Set the Content-Disposition header to force download
    filename = os.path.basename(file_path)
    response['Content-Disposition'] = f'attachment; filename="{filename}"'
    
    return response

# Supplier Note Views
class SupplierNoteCreateView(LoginRequiredMixin, CreateView):
    model = SupplierNote
    form_class = SupplierNoteForm
    template_name = 'suppliers/note_form.html'
    
    def form_valid(self, form):
        form.instance.supplier_id = self.kwargs['supplier_pk']
        form.instance.created_by = self.request.user
        messages.success(self.request, 'Note added successfully.')
        return super().form_valid(form)
    
    def get_success_url(self):
        return reverse('suppliers:supplier_detail', kwargs={'pk': self.kwargs['supplier_pk']})
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['supplier'] = get_object_or_404(Supplier, pk=self.kwargs['supplier_pk'])
        context['title'] = 'Add Note'
        return context

class SupplierNoteUpdateView(LoginRequiredMixin, UpdateView):
    model = SupplierNote
    form_class = SupplierNoteForm
    template_name = 'suppliers/note_form.html'
    
    def form_valid(self, form):
        messages.success(self.request, 'Note updated successfully.')
        return super().form_valid(form)
    
    def get_success_url(self):
        return reverse('suppliers:supplier_detail', kwargs={'pk': self.object.supplier.pk})
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['supplier'] = self.object.supplier
        context['title'] = 'Update Note'
        return context

class SupplierNoteDeleteView(LoginRequiredMixin, DeleteView):
    model = SupplierNote
    template_name = 'suppliers/note_confirm_delete.html'
    
    def test_func(self):
        note = self.get_object()
        return self.request.user.is_staff or self.request.user == note.created_by
    
    def delete(self, request, *args, **kwargs):
        note = self.get_object()
        supplier_pk = note.supplier.pk
        note.delete()
        messages.success(request, 'Note deleted successfully.')
        return HttpResponseRedirect(reverse('suppliers:supplier_detail', kwargs={'pk': supplier_pk}))
    
    def get_success_url(self):
        return reverse('suppliers:supplier_detail', kwargs={'pk': self.object.supplier.pk})

# AJAX Views
@login_required
@require_http_methods(['POST'])
def toggle_supplier_status(request, pk):
    supplier = get_object_or_404(Supplier, pk=pk)
    supplier.is_active = not supplier.is_active
    supplier.save()
    
    status = 'active' if supplier.is_active else 'inactive'
    return JsonResponse({
        'status': 'success',
        'is_active': supplier.is_active,
        'message': f'Supplier is now {status}.'
    })

@login_required
def supplier_autocomplete(request):
    query = request.GET.get('q', '')
    suppliers = Supplier.objects.filter(
        Q(name__icontains=query) |
        Q(company_name__icontains=query) |
        Q(email__icontains=query) |
        Q(phone__icontains=query)
    ).filter(is_active=True)
    
    results = [{
        'id': supplier.id,
        'text': f"{supplier.name} ({supplier.company_name or 'No Company'}) - {supplier.phone or 'No Phone'}",
        'name': supplier.name,
        'company': supplier.company_name,
        'email': supplier.email,
        'phone': supplier.phone,
        'address': supplier.get_full_address()
    } for supplier in suppliers[:10]]
    
    return JsonResponse({'results': results})

def supplier_detail_api(request, pk):
    """API endpoint to get supplier details."""
    supplier = get_object_or_404(Supplier, pk=pk)
    data = {
        'id': supplier.id,
        'name': supplier.name,
        'company_name': supplier.company_name,
        'email': supplier.email,
        'phone': supplier.phone,
        'address': f"{supplier.address_line1}, {supplier.city}, {supplier.country}",
        'credit_limit': str(supplier.credit_limit),
        'balance': str(supplier.balance),
        'available_credit': str(supplier.credit_limit - supplier.balance) if supplier.credit_limit else 'Unlimited'
    }
    return JsonResponse(data)

# Supplier Category Views
class SupplierCategoryListView(LoginRequiredMixin, ListView):
    model = SupplierCategory
    template_name = 'suppliers/category_list.html'
    context_object_name = 'categories'
    paginate_by = 20
    
    def get_queryset(self):
        queryset = SupplierCategory.objects.all()
        
        # Filter by status
        status = self.request.GET.get('status')
        if status == 'active':
            queryset = queryset.filter(is_active=True)
        elif status == 'inactive':
            queryset = queryset.filter(is_active=False)
        
        # Search functionality
        search_query = self.request.GET.get('q')
        if search_query:
            queryset = queryset.filter(
                Q(name__icontains=search_query) |
                Q(description__icontains=search_query)
            )
        
        return queryset

class SupplierCategoryCreateView(LoginRequiredMixin, CreateView):
    model = SupplierCategory
    form_class = SupplierCategoryForm
    template_name = 'suppliers/category_form.html'
    success_url = reverse_lazy('suppliers:supplier_category_list')
    
    def form_valid(self, form):
        response = super().form_valid(form)
        messages.success(self.request, f'Category "{self.object.name}" was created successfully.')
        return response

class SupplierCategoryUpdateView(LoginRequiredMixin, UpdateView):
    model = SupplierCategory
    form_class = SupplierCategoryForm
    template_name = 'suppliers/category_form.html'
    success_url = reverse_lazy('suppliers:supplier_category_list')
    
    def form_valid(self, form):
        response = super().form_valid(form)
        messages.success(self.request, f'Category "{self.object.name}" was updated successfully.')
        return response

class SupplierCategoryDeleteView(LoginRequiredMixin, DeleteView):
    model = SupplierCategory
    template_name = 'suppliers/category_confirm_delete.html'
    success_url = reverse_lazy('suppliers:supplier_category_list')
    
    def test_func(self):
        return self.request.user.is_staff
    
    def delete(self, request, *args, **kwargs):
        category = self.get_object()
        
        # Check if there are suppliers in this category
        if category.suppliers.exists():
            messages.error(request, f'Cannot delete "{category.name}" because it has associated suppliers.')
            return HttpResponseRedirect(self.success_url)
        
        response = super().delete(request, *args, **kwargs)
        messages.success(request, f'Category "{category.name}" was deleted successfully.')
        return response
