from django import forms
from .models import Category, Product

class CategoryForm(forms.ModelForm):
    """
    Form for category management
    """
    class Meta:
        model = Category
        fields = ['name', 'description', 'branch']
        widgets = {
            'name': forms.TextInput(attrs={'class': 'form-control'}),
            'description': forms.Textarea(attrs={'class': 'form-control', 'rows': 3}),
            'branch': forms.Select(attrs={'class': 'form-control'}),
        }

class ProductForm(forms.ModelForm):
    """
    Form for product management
    """
    class Meta:
        model = Product
        fields = [
            'name', 'description', 'category', 'branch', 
            'cost_price', 'price', 'quantity', 'low_stock_threshold', 
            'sku', 'barcode', 'image', 'is_active'
        ]
        widgets = {
            'name': forms.TextInput(attrs={'class': 'form-control'}),
            'description': forms.Textarea(attrs={'class': 'form-control', 'rows': 3}),
            'category': forms.Select(attrs={'class': 'form-control'}),
            'branch': forms.Select(attrs={'class': 'form-control'}),
            'cost_price': forms.NumberInput(attrs={
                'class': 'form-control', 
                'step': '0.01',
                'required': 'required',
                'min': '0.01',
                'placeholder': 'Enter purchase price'
            }),
            'price': forms.NumberInput(attrs={
                'class': 'form-control', 
                'step': '0.01',
                'required': 'required',
                'min': '0.01',
                'placeholder': 'Enter selling price'
            }),
            'quantity': forms.NumberInput(attrs={
                'class': 'form-control',
                'min': '0',
                'step': '1',
                'required': 'required'
            }),
            'low_stock_threshold': forms.NumberInput(attrs={
                'class': 'form-control',
                'min': '1',
                'step': '1',
                'required': 'required',
                'value': '10'  # Default value
            }),
            'sku': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': 'e.g., PROD-001'
            }),
            'barcode': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': 'Scan or enter barcode'
            }),
            'image': forms.FileInput(attrs={
                'class': 'form-control', 
                'accept': 'image/*',
                'onchange': 'previewImage(this)'
            }),
            'is_active': forms.CheckboxInput(attrs={
                'class': 'form-check-input',
                'checked': 'checked'
            }),
        }
        help_texts = {
            'cost_price': 'The price you pay to purchase this item.',
            'price': 'The price at which you sell this item to customers.',
            'low_stock_threshold': 'Get notified when stock falls below this number.'
        }
    
    def clean_cost_price(self):
        """Validate and ensure cost_price is valid"""
        cost_price = self.cleaned_data.get('cost_price', 0)
        if cost_price is None or cost_price < 0:
            raise forms.ValidationError('Purchase price must be zero or greater')
        return cost_price
    
    def clean_price(self):
        """Validate selling price is valid and not less than cost price"""
        price = self.cleaned_data.get('price')
        cost_price = self.cleaned_data.get('cost_price', 0)
        
        if price is None or price <= 0:
            raise forms.ValidationError('Selling price must be greater than zero')
            
        if price < cost_price:
            raise forms.ValidationError('Selling price should not be less than purchase price')
            
        return price

class ProductSearchForm(forms.Form):
    """
    Form for searching products
    """
    query = forms.CharField(
        required=False,
        widget=forms.TextInput(attrs={
            'class': 'form-control',
            'placeholder': 'Search by name, SKU, or barcode'
        })
    )
    category = forms.ModelChoiceField(
        queryset=Category.objects.none(),
        required=False,
        empty_label="All Categories",
        widget=forms.Select(attrs={'class': 'form-control'})
    )
    
    def __init__(self, user, *args, **kwargs):
        super().__init__(*args, **kwargs)
        
        # Filter categories by user's branch or show all for admin
        if user.role == 'admin':
            self.fields['category'].queryset = Category.objects.all()
        else:
            self.fields['category'].queryset = Category.objects.filter(branch=user.branch)

class StockAdjustmentForm(forms.Form):
    """
    Form for adjusting product stock
    """
    ADJUSTMENT_TYPES = [
        ('add', 'Add to Stock'),
        ('remove', 'Remove from Stock'),
        ('set', 'Set Stock Level'),
    ]
    
    adjustment_type = forms.ChoiceField(
        choices=ADJUSTMENT_TYPES,
        widget=forms.Select(attrs={'class': 'form-control'})
    )
    quantity = forms.IntegerField(
        widget=forms.NumberInput(attrs={'class': 'form-control', 'min': '0'})
    )
    reason = forms.CharField(
        widget=forms.Textarea(attrs={'class': 'form-control', 'rows': 3}),
        help_text="Provide a reason for this stock adjustment."
    )
