from django.db import models
from django.utils.translation import gettext_lazy as _
from django.contrib.auth import get_user_model
from django.utils import timezone
from django.core.validators import MinValueValidator
from decimal import Decimal

from branches.models import Branch

User = get_user_model()

class ExpenseCategory(models.Model):
    """Model for categorizing expenses"""
    name = models.CharField(_('category name'), max_length=100, unique=True)
    description = models.TextField(_('description'), blank=True)
    is_active = models.BooleanField(_('active'), default=True)
    created_at = models.DateTimeField(_('created at'), auto_now_add=True)
    updated_at = models.DateTimeField(_('updated at'), auto_now=True)

    class Meta:
        verbose_name = _('expense category')
        verbose_name_plural = _('expense categories')
        ordering = ['name']

    def __str__(self):
        return self.name


class Expense(models.Model):
    """Model for tracking business expenses"""
    
    class ExpenseStatus(models.TextChoices):
        DRAFT = 'draft', _('Draft')
        PENDING = 'pending', _('Pending Approval')
        APPROVED = 'approved', _('Approved')
        REJECTED = 'rejected', _('Rejected')
        PAID = 'paid', _('Paid')
        
    class PaymentMethod(models.TextChoices):
        CASH = 'cash', _('Cash')
        BANK_TRANSFER = 'bank_transfer', _('Bank Transfer')
        CHEQUE = 'cheque', _('Cheque')
        CREDIT_CARD = 'credit_card', _('Credit Card')
        MOBILE_MONEY = 'mobile_money', _('Mobile Money')
        OTHER = 'other', _('Other')
    
    reference = models.CharField(_('reference number'), max_length=50, unique=True)
    category = models.ForeignKey(
        ExpenseCategory, 
        on_delete=models.PROTECT, 
        related_name='expenses',
        verbose_name=_('category')
    )
    branch = models.ForeignKey(
        Branch, 
        on_delete=models.PROTECT, 
        related_name='expenses',
        verbose_name=_('branch')
    )
    amount = models.DecimalField(
        _('amount'), 
        max_digits=12, 
        decimal_places=2,
        validators=[MinValueValidator(Decimal('0.01'))]
    )
    tax_amount = models.DecimalField(
        _('tax amount'), 
        max_digits=12, 
        decimal_places=2,
        default=0.00,
        validators=[MinValueValidator(Decimal('0.00'))]
    )
    total_amount = models.DecimalField(
        _('total amount'), 
        max_digits=12, 
        decimal_places=2,
        validators=[MinValueValidator(Decimal('0.01'))]
    )
    expense_date = models.DateField(_('expense date'), default=timezone.now)
    payment_date = models.DateField(_('payment date'), null=True, blank=True)
    payment_method = models.CharField(
        _('payment method'), 
        max_length=20, 
        choices=PaymentMethod.choices,
        default=PaymentMethod.CASH
    )
    payment_reference = models.CharField(
        _('payment reference'), 
        max_length=100, 
        blank=True
    )
    status = models.CharField(
        _('status'), 
        max_length=20, 
        choices=ExpenseStatus.choices, 
        default=ExpenseStatus.DRAFT
    )
    description = models.TextField(_('description'), blank=True)
    receipt = models.FileField(
        _('receipt'), 
        upload_to='expense_receipts/%Y/%m/%d/', 
        null=True, 
        blank=True
    )
    
    # Audit fields
    created_by = models.ForeignKey(
        User, 
        on_delete=models.PROTECT, 
        related_name='created_expenses',
        verbose_name=_('created by')
    )
    approved_by = models.ForeignKey(
        User, 
        on_delete=models.SET_NULL, 
        null=True, 
        blank=True,
        related_name='approved_expenses',
        verbose_name=_('approved by')
    )
    approved_at = models.DateTimeField(_('approved at'), null=True, blank=True)
    created_at = models.DateTimeField(_('created at'), auto_now_add=True)
    updated_at = models.DateTimeField(_('updated at'), auto_now=True)

    class Meta:
        verbose_name = _('expense')
        verbose_name_plural = _('expenses')
        ordering = ['-expense_date', '-created_at']
        permissions = [
            ('approve_expense', 'Can approve expense'),
            ('view_all_expenses', 'Can view all expenses'),
            ('export_expenses', 'Can export expense data'),
        ]

    def __str__(self):
        return f"{self.reference} - {self.category} - {self.amount}"
    
    def save(self, *args, **kwargs):
        # Auto-calculate total amount if not set
        if not self.total_amount:
            self.total_amount = self.amount + self.tax_amount
        
        # Set payment date when status changes to paid
        if self.status == self.ExpenseStatus.PAID and not self.payment_date:
            self.payment_date = timezone.now().date()
        
        # Set approved_at when status changes to approved
        if self.status == self.ExpenseStatus.APPROVED and not self.approved_at:
            self.approved_at = timezone.now()
        
        super().save(*args, **kwargs)
    
    @property
    def is_approved(self):
        return self.status == self.ExpenseStatus.APPROVED
    
    @property
    def is_paid(self):
        return self.status == self.ExpenseStatus.PAID
    
    def get_absolute_url(self):
        from django.urls import reverse
        return reverse('expenses:expense_detail', kwargs={'pk': self.pk})


class ExpenseItem(models.Model):
    """Model for individual expense line items"""
    expense = models.ForeignKey(
        Expense, 
        on_delete=models.CASCADE, 
        related_name='items',
        verbose_name=_('expense')
    )
    description = models.CharField(_('description'), max_length=255)
    amount = models.DecimalField(
        _('amount'), 
        max_digits=12, 
        decimal_places=2,
        validators=[MinValueValidator(Decimal('0.01'))]
    )
    quantity = models.DecimalField(
        _('quantity'), 
        max_digits=10, 
        decimal_places=2,
        default=1,
        validators=[MinValueValidator(Decimal('0.01'))]
    )
    unit_price = models.DecimalField(
        _('unit price'), 
        max_digits=12, 
        decimal_places=2,
        validators=[MinValueValidator(Decimal('0.01'))]
    )
    tax_rate = models.DecimalField(
        _('tax rate (%)'), 
        max_digits=5, 
        decimal_places=2,
        default=0.00,
        validators=[MinValueValidator(Decimal('0.00')), MinValueValidator(Decimal('100.00'))]
    )
    tax_amount = models.DecimalField(
        _('tax amount'), 
        max_digits=12, 
        decimal_places=2,
        default=0.00
    )
    total_amount = models.DecimalField(
        _('total amount'), 
        max_digits=12, 
        decimal_places=2
    )
    created_at = models.DateTimeField(_('created at'), auto_now_add=True)
    updated_at = models.DateTimeField(_('updated at'), auto_now=True)

    class Meta:
        verbose_name = _('expense item')
        verbose_name_plural = _('expense items')
        ordering = ['id']

    def __str__(self):
        return f"{self.description[:50]} - {self.amount}"
    
    def save(self, *args, **kwargs):
        # Calculate amounts if not set
        if not self.unit_price and self.quantity and self.amount:
            self.unit_price = self.amount / self.quantity
        
        if not self.amount and self.unit_price and self.quantity:
            self.amount = self.unit_price * self.quantity
        
        # Calculate tax and total
        if self.tax_rate > 0:
            self.tax_amount = (self.amount * self.tax_rate) / 100
        else:
            self.tax_amount = Decimal('0.00')
        
        self.total_amount = self.amount + self.tax_amount
        
        super().save(*args, **kwargs)
        
        # Update parent expense amounts
        self.expense.amount = self.expense.items.aggregate(
            total=models.Sum('amount')
        )['total'] or Decimal('0.00')
        
        self.expense.tax_amount = self.expense.items.aggregate(
            total=models.Sum('tax_amount')
        )['total'] or Decimal('0.00')
        
        self.expense.total_amount = self.expense.items.aggregate(
            total=models.Sum('total_amount')
        )['total'] or Decimal('0.00')
        
        self.expense.save()
