from django import forms
from django.forms import ModelForm
from django.utils import timezone
from django.utils.translation import gettext_lazy as _
from .models import Expense, ExpenseCategory

class ExpenseForm(ModelForm):
    class Meta:
        model = Expense
        fields = ['description', 'category', 'amount', 'expense_date', 'payment_method', 'notes', 'status']
        widgets = {
            'description': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Enter expense description')
            }),
            'category': forms.Select(attrs={
                'class': 'form-select',
            }),
            'amount': forms.NumberInput(attrs={
                'class': 'form-control',
                'step': '0.01',
                'min': '0.01'
            }),
            'expense_date': forms.DateInput(attrs={
                'class': 'form-control',
                'type': 'date'
            }),
            'payment_method': forms.Select(attrs={
                'class': 'form-select',
            }),
            'notes': forms.Textarea(attrs={
                'class': 'form-control',
                'rows': 3,
                'placeholder': _('Any additional notes...')
            }),
            'status': forms.Select(attrs={
                'class': 'form-select',
            }),
        }

    def __init__(self, *args, **kwargs):
        self.request = kwargs.pop('request', None)
        super().__init__(*args, **kwargs)
        
        # Set default status to approved
        self.fields['status'].initial = 'approved'
        
        # Set default date to today
        self.fields['expense_date'].initial = timezone.now().date()
        
        # Set created_by to current user
        if self.request and self.request.user.is_authenticated:
            self.fields['created_by'] = forms.IntegerField(
                widget=forms.HiddenInput(),
                initial=self.request.user.id
            )
        
        # Update category queryset to only show active categories
        self.fields['category'].queryset = ExpenseCategory.objects.filter(is_active=True)
        
        # Add a new category field
        self.fields['new_category'] = forms.CharField(
            required=False,
            widget=forms.TextInput(attrs={
                'class': 'form-control mt-2',
                'placeholder': _('Enter new category name')
            })
        )

class ExpenseCategoryForm(ModelForm):
    class Meta:
        model = ExpenseCategory
        fields = ['name', 'description', 'is_active']
        widgets = {
            'name': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Category name')
            }),
            'description': forms.Textarea(attrs={
                'class': 'form-control',
                'rows': 2,
                'placeholder': _('Optional description')
            }),
            'is_active': forms.CheckboxInput(attrs={
                'class': 'form-check-input'
            })
        }
