from django.db import models
from django.utils import timezone
from datetime import timedelta
import hashlib
import secrets
import string
import base64


class License(models.Model):
    """
    License model for managing access to branches
    """
    license_key = models.CharField(max_length=100, unique=True)
    branch = models.ForeignKey('branches.Branch', on_delete=models.CASCADE, related_name='licenses')
    issue_date = models.DateTimeField(default=timezone.now)
    expiry_date = models.DateTimeField()
    is_active = models.BooleanField(default=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    def __str__(self):
        return self.license_key
    
    @property
    def is_expired(self):
        """
        Check if license has expired
        """
        return self.expiry_date < timezone.now()
    
    @property
    def days_remaining(self):
        """
        Calculate number of days remaining until expiry
        """
        if self.is_expired:
            return 0
        
        delta = self.expiry_date - timezone.now()
        return max(0, delta.days)
    
    @staticmethod
    def generate_license_key(branch_id, days=365):
        """
        Generate a new license key for a branch
        
        Format: XXXXX-XXXXX-XXXXX-XXXXX-XXXXX
        where X is alphanumeric
        
        The key is generated using a combination of:
        - Branch ID
        - Current timestamp
        - Random data
        - Checksum
        """
        # Generate timestamp and expiry date
        issue_date = timezone.now()
        expiry_date = issue_date + timedelta(days=days)
        
        # Generate seed data
        timestamp = int(issue_date.timestamp())
        random_string = ''.join(secrets.choice(string.ascii_uppercase + string.digits) for _ in range(8))
        
        # Create raw key data
        raw_data = f"{branch_id}:{timestamp}:{random_string}"
        
        # Create a hash
        hash_obj = hashlib.sha256(raw_data.encode())
        hash_digest = hash_obj.digest()
        
        # Convert to base64 and remove non-alphanumeric
        b64_str = base64.b64encode(hash_digest).decode('utf-8')
        alphanumeric = ''.join(c for c in b64_str if c.isalnum()).upper()
        
        # Format as XXXXX-XXXXX-XXXXX-XXXXX-XXXXX
        chunks = [alphanumeric[i:i+5] for i in range(0, min(25, len(alphanumeric)), 5)]
        license_key = '-'.join(chunks)
        
        return {
            'license_key': license_key,
            'issue_date': issue_date,
            'expiry_date': expiry_date
        }
    
    @staticmethod
    def verify_license_key(license_key):
        """
        Verify a license key format
        """
        # Check format (XXXXX-XXXXX-XXXXX-XXXXX-XXXXX)
        parts = license_key.split('-')
        if len(parts) != 5:
            return False
        
        for part in parts:
            if len(part) != 5 or not all(c.isalnum() for c in part):
                return False
        
        return True