from django import forms
from django.utils import timezone
from .models import Sale, Payment, MpesaPayment
from customers.models import Customer

class SaleForm(forms.ModelForm):
    """
    Form for sale details
    """
    created_at = forms.DateTimeField(
        required=False,
        widget=forms.DateTimeInput(attrs={'class': 'form-control', 'type': 'datetime-local'}),
        label='Transaction Date & Time',
        help_text='Set a custom date and time for this sale (Kenya time). Leave blank for now.'
    )
    class Meta:
        model = Sale
        fields = ['customer', 'notes', 'created_at']
        widgets = {
            'customer': forms.Select(attrs={'class': 'form-control'}),
            'notes': forms.Textarea(attrs={'class': 'form-control', 'rows': 3}),
        }

class QuickCustomerForm(forms.ModelForm):
    """
    Form for quick customer creation during sale
    """
    class Meta:
        model = Customer
        fields = ['first_name', 'last_name', 'email', 'phone', 'branch']
        widgets = {
            'first_name': forms.TextInput(attrs={'class': 'form-control'}),
            'last_name': forms.TextInput(attrs={'class': 'form-control'}),
            'email': forms.EmailInput(attrs={'class': 'form-control'}),
            'phone': forms.TextInput(attrs={'class': 'form-control'}),
            'branch': forms.HiddenInput(),
        }

class PaymentForm(forms.ModelForm):
    """
    Form for adding payments
    """
    mpesa_phone = forms.CharField(
        max_length=15,
        required=False,
        widget=forms.TextInput(attrs={
            'class': 'form-control',
            'placeholder': 'Phone number for M-Pesa'
        }),
        help_text="Required for M-Pesa payments"
    )
    
    class Meta:
        model = Payment
        fields = ['amount', 'payment_method', 'reference_number', 'notes']
        widgets = {
            'amount': forms.NumberInput(attrs={'class': 'form-control', 'step': '0.01'}),
            'payment_method': forms.Select(attrs={'class': 'form-control'}),
            'reference_number': forms.TextInput(attrs={'class': 'form-control'}),
            'notes': forms.Textarea(attrs={'class': 'form-control', 'rows': 3}),
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['reference_number'].help_text = "Transaction code for M-Pesa payments"
    
    def clean(self):
        cleaned_data = super().clean()
        payment_method = cleaned_data.get('payment_method')
        reference_number = cleaned_data.get('reference_number')
        mpesa_phone = cleaned_data.get('mpesa_phone')
        
        # Validate M-Pesa specific fields
        if payment_method == 'mpesa':
            if not reference_number:
                self.add_error('reference_number', 'Transaction code is required for M-Pesa payments')
            if not mpesa_phone:
                self.add_error('mpesa_phone', 'Phone number is required for M-Pesa payments')
        
        return cleaned_data
    
    def save(self, commit=True):
        payment = super().save(commit=commit)
        
        if commit and payment.payment_method == 'mpesa' and hasattr(self, 'cleaned_data'):
            # Create M-Pesa payment record
            MpesaPayment.objects.create(
                payment=payment,
                phone_number=self.cleaned_data.get('mpesa_phone'),
                transaction_id=payment.reference_number,
                status='completed'  # For direct entry, assume it's already completed
            )
        
        return payment

class SaleSearchForm(forms.Form):
    """
    Form for searching and filtering sales
    """
    start_date = forms.DateField(
        required=False,
        widget=forms.DateInput(attrs={
            'class': 'form-control',
            'type': 'date'
        })
    )
    end_date = forms.DateField(
        required=False,
        widget=forms.DateInput(attrs={
            'class': 'form-control',
            'type': 'date'
        })
    )
    customer = forms.CharField(
        required=False,
        widget=forms.TextInput(attrs={
            'class': 'form-control',
            'placeholder': 'Customer name'
        })
    )
    invoice = forms.CharField(
        required=False,
        widget=forms.TextInput(attrs={
            'class': 'form-control',
            'placeholder': 'Invoice number'
        })
    )
    payment_status = forms.ChoiceField(
        required=False,
        choices=[('', 'All')] + list(Sale.PAYMENT_STATUS_CHOICES),
        widget=forms.Select(attrs={'class': 'form-control'})
    )