from django.db import models
from django.conf import settings
from django.utils import timezone
import uuid

class Sale(models.Model):
    """
    Sale model for storing transaction details
    """
    PAYMENT_STATUS_CHOICES = [
        ('pending', 'Pending'),
        ('partial', 'Partially Paid'),
        ('paid', 'Paid'),
        ('cancelled', 'Cancelled'),
    ]
    
    invoice_number = models.CharField(max_length=50, unique=True)
    customer = models.ForeignKey('customers.Customer', on_delete=models.SET_NULL, null=True, blank=True)
    cashier = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.SET_NULL, null=True, related_name='sales')
    branch = models.ForeignKey('branches.Branch', on_delete=models.CASCADE)
    subtotal = models.DecimalField(max_digits=10, decimal_places=2, default=0)
    tax_amount = models.DecimalField(max_digits=10, decimal_places=2, default=0)
    discount_amount = models.DecimalField(max_digits=10, decimal_places=2, default=0)
    total_amount = models.DecimalField(max_digits=10, decimal_places=2, default=0)
    payment_status = models.CharField(max_length=10, choices=PAYMENT_STATUS_CHOICES, default='pending')
    notes = models.TextField(blank=True)
    created_at = models.DateTimeField(default=timezone.now, blank=True, null=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    class Meta:
        ordering = ['-created_at']
    
    def __str__(self):
        return self.invoice_number
    
    def save(self, *args, **kwargs):
        # Generate invoice number if not provided
        if not self.invoice_number:
            current_date = timezone.now().strftime('%Y%m%d')
            self.invoice_number = f"INV-{current_date}-{uuid.uuid4().hex[:6].upper()}"
        
        # Calculate totals if not explicitly set
        if not self.total_amount:
            self.calculate_totals()
        
        super().save(*args, **kwargs)
    
    def calculate_totals(self):
        """
        Calculate sale totals based on items
        """
        # Get items if the sale has been saved
        if self.pk:
            items = self.saleitems.all()
            self.subtotal = sum(item.total for item in items)
        
        # Calculate tax and total
        self.tax_amount = self.subtotal * (self.branch.tax_rate / 100) if hasattr(self.branch, 'tax_rate') else 0
        self.total_amount = self.subtotal + self.tax_amount - self.discount_amount
        
        return self.total_amount
    
    @property
    def total_paid(self):
        """
        Calculate total amount paid for this sale
        """
        return self.payments.aggregate(total=models.Sum('amount'))['total'] or 0
    
    @property
    def balance_due(self):
        """
        Calculate remaining balance
        """
        return self.total_amount - self.total_paid
    
    def update_payment_status(self):
        """
        Update payment status based on payments
        """
        total_paid = self.total_paid
        
        if total_paid >= self.total_amount:
            self.payment_status = 'paid'
        elif total_paid > 0:
            self.payment_status = 'partial'
        else:
            self.payment_status = 'pending'
        
        self.save(update_fields=['payment_status'])

class SaleItem(models.Model):
    """
    Individual items in a sale
    """
    sale = models.ForeignKey(Sale, on_delete=models.CASCADE, related_name='saleitems')
    product = models.ForeignKey('products.Product', on_delete=models.PROTECT)
    quantity = models.IntegerField(default=1)
    price = models.DecimalField(max_digits=10, decimal_places=2)  # Price at time of sale
    total = models.DecimalField(max_digits=10, decimal_places=2)
    
    def __str__(self):
        return f"{self.product.name} - {self.quantity} x {self.price}"
    
    def save(self, *args, **kwargs):
        # Calculate total
        self.total = self.quantity * self.price
        
        # Save the item
        super().save(*args, **kwargs)
        
        # Update the sale totals
        self.sale.calculate_totals()
        self.sale.save()
        
        # Reduce product stock unless specified otherwise
        if kwargs.pop('update_stock', True):
            self.product.adjust_stock(
                'remove', 
                self.quantity, 
                reason=f"Sale - Invoice #{self.sale.invoice_number}"
            )

class Payment(models.Model):
    """
    Payment records for sales
    """
    PAYMENT_METHOD_CHOICES = [
        ('cash', 'Cash'),
        ('mpesa', 'M-Pesa'),
        ('card', 'Credit/Debit Card'),
        ('transfer', 'Bank Transfer'),
        ('check', 'Check'),
        ('other', 'Other'),
    ]
    
    sale = models.ForeignKey(Sale, on_delete=models.CASCADE, related_name='payments')
    amount = models.DecimalField(max_digits=10, decimal_places=2)
    payment_method = models.CharField(max_length=10, choices=PAYMENT_METHOD_CHOICES, default='cash')
    reference_number = models.CharField(max_length=100, blank=True, help_text="For M-Pesa, enter the transaction code")
    payment_date = models.DateTimeField(default=timezone.now)
    received_by = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.SET_NULL, null=True, blank=True)
    notes = models.TextField(blank=True)
    
    def __str__(self):
        return f"Payment of {self.amount} for {self.sale.invoice_number}"
    
    def save(self, *args, **kwargs):
        super().save(*args, **kwargs)
        
        # Update the sale's payment status
        self.sale.update_payment_status()


class MpesaPayment(models.Model):
    """
    M-Pesa payment details
    """
    STATUS_CHOICES = [
        ('pending', 'Pending'),
        ('completed', 'Completed'),
        ('failed', 'Failed'),
    ]
    
    payment = models.OneToOneField(Payment, on_delete=models.CASCADE, related_name='mpesa_details')
    phone_number = models.CharField(max_length=15)
    transaction_id = models.CharField(max_length=50, unique=True)
    confirmation_code = models.CharField(max_length=50, unique=True, blank=True, null=True)
    status = models.CharField(max_length=10, choices=STATUS_CHOICES, default='pending')
    processed_at = models.DateTimeField(auto_now_add=True)
    
    def __str__(self):
        return f"M-Pesa payment {self.transaction_id} - {self.status}"
