"""
Utility functions for the core application
"""
import logging
from django.core.mail import send_mail, EmailMultiAlternatives
from django.conf import settings
from django.template.loader import render_to_string
from django.utils.html import strip_tags

logger = logging.getLogger('asaploads_pos')

def send_email(subject, recipient_list, template_name, context, from_email=None):
    """
    Send an HTML email using a template
    
    Args:
        subject (str): Email subject
        recipient_list (list): List of email recipients
        template_name (str): Template file path
        context (dict): Context data for the template
        from_email (str, optional): Sender email. Defaults to settings.DEFAULT_FROM_EMAIL.
    
    Returns:
        bool: True if successful, False otherwise
    """
    try:
        if not from_email:
            from_email = settings.DEFAULT_FROM_EMAIL
            
        # Render HTML email template
        html_content = render_to_string(template_name, context)
        text_content = strip_tags(html_content)  # Plain text version
        
        # Create email message
        email = EmailMultiAlternatives(
            subject=subject,
            body=text_content,
            from_email=from_email,
            to=recipient_list
        )
        email.attach_alternative(html_content, "text/html")
        
        # Send email
        sent = email.send()
        if sent:
            logger.info(f"Email sent successfully to {', '.join(recipient_list)}")
            return True
        else:
            logger.error(f"Failed to send email to {', '.join(recipient_list)}")
            return False
    
    except Exception as e:
        logger.error(f"Error sending email: {str(e)}")
        return False

def send_business_credentials_email(email, context):
    """
    Send login credentials to a new business admin
    
    Args:
        email (str): Recipient email
        context (dict): Context data with business and credential details
    
    Returns:
        bool: True if successful, False otherwise
    """
    subject = f"Your AsapLoads POS Account Details for {context.get('branch_name', 'your business')}"
    return send_email(
        subject=subject,
        recipient_list=[email],
        template_name='emails/business_credentials.html',
        context=context
    )
