/* 
 * AsapLoads POS System - Main JavaScript
 */

document.addEventListener('DOMContentLoaded', function() {
    // Initialize tooltips
    initializeTooltips();
    
    // Initialize sidebar toggle
    initializeSidebar();
    
    // Add responsive table wrapper
    makeTablesResponsive();
    
    // Initialize any charts on the page
    initializeCharts();
    
    // Add form validation
    validateForms();
    
    // Confirmation for delete actions
    confirmDelete();
});

/**
 * Initialize Bootstrap tooltips
 */
function initializeTooltips() {
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
}

/**
 * Initialize sidebar functionality
 */
function initializeSidebar() {
    const sidebarToggle = document.querySelector('.sidebar-toggle');
    const sidebar = document.querySelector('.sidebar');
    const body = document.body;
    
    if (sidebarToggle && sidebar) {
        const isMobile = window.innerWidth < 992;
        
        // Initialize sidebar state
        if (isMobile) {
            body.classList.add('sidebar-collapsed');
        }
        
        // Toggle sidebar function
        const toggleSidebar = () => {
            if (isMobile) {
                // On mobile, toggle the visibility class
                body.classList.toggle('sidebar-visible');
                
                // Prevent body scroll when sidebar is open on mobile
                if (body.classList.contains('sidebar-visible')) {
                    body.style.overflow = 'hidden';
                } else {
                    body.style.overflow = '';
                }
            } else {
                // On desktop, toggle the collapsed class
                body.classList.toggle('sidebar-collapsed');
                
                // Save the collapsed state for desktop
                localStorage.setItem('sidebarCollapsed', body.classList.contains('sidebar-collapsed'));
            }
            
            // Dispatch a custom event for any other components that need to know
            document.dispatchEvent(new CustomEvent('sidebarToggle', {
                detail: { isCollapsed: body.classList.contains('sidebar-collapsed') }
            }));
        };
        
        // Add click event to toggle button
        sidebarToggle.addEventListener('click', (e) => {
            e.stopPropagation();
            toggleSidebar();
        });
        
        // Close sidebar when clicking outside on mobile
        document.addEventListener('click', (e) => {
            if (isMobile && 
                !sidebar.contains(e.target) && 
                !e.target.closest('.sidebar-toggle') &&
                body.classList.contains('sidebar-visible')) {
                body.classList.remove('sidebar-visible');
                body.style.overflow = '';
            }
        });
        
        // Close sidebar when pressing Escape key
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape' && body.classList.contains('sidebar-visible')) {
                body.classList.remove('sidebar-visible');
                body.style.overflow = '';
            }
        });
        
        // Handle window resize
        let resizeTimer;
        const handleResize = () => {
            clearTimeout(resizeTimer);
            resizeTimer = setTimeout(() => {
                const newIsMobile = window.innerWidth < 992;
                if (isMobile !== newIsMobile) {
                    if (newIsMobile) {
                        body.classList.add('sidebar-collapsed');
                        body.classList.remove('sidebar-visible');
                    } else {
                        body.classList.remove('sidebar-visible');
                        body.style.overflow = '';
                    }
                }
            }, 100);
        };
        
        window.addEventListener('resize', handleResize);
        
        // Clean up event listeners when component is destroyed
        return () => {
            window.removeEventListener('resize', handleResize);
        };
    }
}

/**
 * Make tables responsive by adding a wrapper
 */
function makeTablesResponsive() {
    const tables = document.querySelectorAll('table:not(.table-responsive)');
    tables.forEach(table => {
        // Check if table is not already in a responsive wrapper
        if (!table.parentElement.classList.contains('table-responsive')) {
            const wrapper = document.createElement('div');
            wrapper.classList.add('table-responsive');
            table.parentNode.insertBefore(wrapper, table);
            wrapper.appendChild(table);
        }
    });
}

/**
 * Initialize charts on the page
 */
function initializeCharts() {
    // Chart.js is already included in the base template
    // This function can be extended for specific chart initializations
}

/**
 * Add form validation
 */
function validateForms() {
    const forms = document.querySelectorAll('.needs-validation');
    
    Array.from(forms).forEach(form => {
        form.addEventListener('submit', event => {
            if (!form.checkValidity()) {
                event.preventDefault();
                event.stopPropagation();
            }
            
            form.classList.add('was-validated');
        }, false);
    });
}

/**
 * Add confirmation for delete actions
 */
function confirmDelete() {
    const deleteButtons = document.querySelectorAll('[data-confirm]');
    
    deleteButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            const message = this.dataset.confirm || 'Are you sure you want to delete this item?';
            
            if (!confirm(message)) {
                e.preventDefault();
            }
        });
    });
}

/**
 * Format currency values
 * @param {number} value - The value to format
 * @param {string} currency - Currency symbol
 * @returns {string} - Formatted currency string
 */
function formatCurrency(value, currency = '$') {
    return currency + parseFloat(value).toFixed(2);
}

/**
 * Format date values
 * @param {string|Date} date - Date string or object
 * @param {string} format - Format string ('short', 'medium', 'long')
 * @returns {string} - Formatted date string
 */
function formatDate(date, format = 'medium') {
    const dateObj = new Date(date);
    
    switch(format) {
        case 'short':
            return dateObj.toLocaleDateString();
        case 'long':
            return dateObj.toLocaleDateString() + ' ' + dateObj.toLocaleTimeString();
        case 'medium':
        default:
            return dateObj.toLocaleDateString() + ' ' + 
                   dateObj.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
    }
}

/**
 * Show alert message
 * @param {string} message - The message to display
 * @param {string} type - Alert type ('success', 'danger', 'warning', 'info')
 * @param {number} duration - Duration in milliseconds (0 for no auto-close)
 */
function showAlert(message, type = 'info', duration = 5000) {
    const alertContainer = document.querySelector('.alert-container') || 
                          createAlertContainer();
    
    const alert = document.createElement('div');
    alert.className = `alert alert-${type} alert-dismissible fade show`;
    alert.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    `;
    
    alertContainer.appendChild(alert);
    
    if (duration > 0) {
        setTimeout(() => {
            alert.classList.remove('show');
            setTimeout(() => alert.remove(), 150);
        }, duration);
    }
}

/**
 * Create alert container if it doesn't exist
 * @returns {HTMLElement} - The alert container
 */
function createAlertContainer() {
    const container = document.createElement('div');
    container.className = 'alert-container position-fixed top-0 end-0 p-3';
    container.style.zIndex = '5000';
    document.body.appendChild(container);
    return container;
}
