from django import forms
from django.forms import inlineformset_factory
from django.forms.models import BaseInlineFormSet
from customers.models import Customer
from .models import Quotation, QuotationItem, QuotationDocument, QuotationNote
from django.utils import timezone

class QuotationForm(forms.ModelForm):
    customer = forms.ModelChoiceField(
        queryset=Customer.objects.all(),
        required=True,
        label='Customer',
        widget=forms.Select(attrs={'class': 'form-control'})
    )
    
    status = forms.ChoiceField(
        choices=Quotation.STATUS_CHOICES,
        required=True,
        widget=forms.Select(attrs={'class': 'form-control'})
    )
    
    class Meta:
        model = Quotation
        fields = ['customer', 'status', 'issue_date', 'expiry_date', 'notes', 'terms']
        widgets = {
            'issue_date': forms.DateInput(
                attrs={
                    'type': 'date', 
                    'class': 'form-control',
                    'placeholder': 'Select issue date'
                }, 
                format='%Y-%m-%d'
            ),
            'expiry_date': forms.DateInput(
                attrs={
                    'type': 'date', 
                    'class': 'form-control',
                    'placeholder': 'Select expiry date'
                }, 
                format='%Y-%m-%d'
            ),
            'notes': forms.Textarea(attrs={
                'rows': 3, 
                'class': 'form-control',
                'placeholder': 'Enter any additional notes'
            }),
            'terms': forms.Textarea(attrs={
                'rows': 3, 
                'class': 'form-control',
                'placeholder': 'Enter terms and conditions'
            }),
        }
    
    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        super().__init__(*args, **kwargs)
        
        # Set default issue date to today
        if not self.instance.pk:
            self.initial['issue_date'] = timezone.now().date()
            
        # Set default expiry date to 30 days from now
        if not self.instance.expiry_date:
            self.initial['expiry_date'] = timezone.now().date() + timezone.timedelta(days=30)

class QuotationItemForm(forms.ModelForm):
    class Meta:
        model = QuotationItem
        fields = ['product', 'quantity', 'unit_price', 'notes']
        widgets = {
            'quantity': forms.NumberInput(attrs={'min': '0.01', 'step': '0.01'}),
            'unit_price': forms.NumberInput(attrs={'min': '0.01', 'step': '0.01'}),
            'notes': forms.Textarea(attrs={'rows': 1}),
        }
    
    def __init__(self, *args, **kwargs):
        user = kwargs.pop('user', None)
        super().__init__(*args, **kwargs)
        
        # Filter products based on user's branch if needed
        if user and hasattr(user, 'branch'):
            self.fields['product'].queryset = self.fields['product'].queryset.filter(
                branch=user.branch
            )

class QuotationItemFormSet(BaseInlineFormSet):
    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        super().__init__(*args, **kwargs)
    
    def _construct_form(self, i, **kwargs):
        kwargs['user'] = self.user
        return super()._construct_form(i, **kwargs)

# Create formset for quotation items
QuotationItemFormSet = inlineformset_factory(
    Quotation,
    QuotationItem,
    form=QuotationItemForm,
    formset=QuotationItemFormSet,
    extra=1,
    can_delete=True,
)

class QuotationDocumentForm(forms.ModelForm):
    class Meta:
        model = QuotationDocument
        fields = ['document', 'notes']
        widgets = {
            'notes': forms.Textarea(attrs={'rows': 2}),
        }

class QuotationNoteForm(forms.ModelForm):
    class Meta:
        model = QuotationNote
        fields = ['note']
        widgets = {
            'note': forms.Textarea(attrs={'rows': 3, 'placeholder': 'Add a note about this quotation...'}),
        }
