from django.shortcuts import render, get_object_or_404, redirect
from django.views.generic import ListView, CreateView, UpdateView, DetailView, DeleteView
from django.urls import reverse_lazy
from django.contrib import messages
from django.contrib.auth.mixins import LoginRequiredMixin, UserPassesTestMixin
from django.utils import timezone
from django.db import transaction

from .models import PurchaseOrder, PurchaseOrderItem, GoodsReceipt, GoodsReceiptItem
from .forms import (
    PurchaseOrderForm, PurchaseOrderItemForm, BasePurchaseOrderItemFormSet,
    GoodsReceiptForm, get_goods_receipt_item_formset, PurchaseOrderItemFormSet
)
from products.models import Product

class PurchaseOrderListView(LoginRequiredMixin, ListView):
    model = PurchaseOrder
    template_name = 'purchases/purchaseorder_list.html'
    context_object_name = 'purchase_orders'
    paginate_by = 20
    
    def get_queryset(self):
        queryset = PurchaseOrder.objects.all()
        
        # Filter by status if provided
        status = self.request.GET.get('status')
        if status in dict(PurchaseOrder.STATUS_CHOICES).keys():
            queryset = queryset.filter(status=status)
            
        # Filter by supplier if provided
        supplier_id = self.request.GET.get('supplier')
        if supplier_id:
            queryset = queryset.filter(supplier_id=supplier_id)
            
        # Search by PO number
        search = self.request.GET.get('search')
        if search:
            queryset = queryset.filter(po_number__icontains=search)
            
        return queryset.order_by('-order_date', '-created_at')
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['status_choices'] = dict(PurchaseOrder.STATUS_CHOICES)
        context['current_status'] = self.request.GET.get('status', '')
        context['search_query'] = self.request.GET.get('search', '')
        return context

class PurchaseOrderCreateView(LoginRequiredMixin, CreateView):
    model = PurchaseOrder
    form_class = PurchaseOrderForm
    template_name = 'purchases/purchaseorder_form.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        if self.request.POST:
            context['formset'] = PurchaseOrderItemFormSet(self.request.POST, prefix='items')
        else:
            context['formset'] = PurchaseOrderItemFormSet(prefix='items')
        return context
    
    def form_valid(self, form):
        context = self.get_context_data()
        formset = context['formset']
        
        with transaction.atomic():
            form.instance.created_by = self.request.user
            self.object = form.save()
            
            if formset.is_valid():
                formset.instance = self.object
                formset.save()
                
                # Update product stock for each item
                for item in self.object.items.all():
                    if item.quantity > 0:
                        item.product.stock += item.quantity
                        item.product.save()
                
                messages.success(self.request, 'Purchase order created successfully.')
                return redirect('purchase_detail', pk=self.object.pk)
            
        return self.render_to_response(self.get_context_data(form=form))

class PurchaseOrderUpdateView(LoginRequiredMixin, UserPassesTestMixin, UpdateView):
    model = PurchaseOrder
    form_class = PurchaseOrderForm
    template_name = 'purchases/purchaseorder_form.html'
    
    def test_func(self):
        purchase_order = self.get_object()
        return self.request.user == purchase_order.created_by or self.request.user.is_superuser
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        if self.request.POST:
            context['formset'] = PurchaseOrderItemFormSet(
                self.request.POST, instance=self.object, prefix='items'
            )
        else:
            context['formset'] = PurchaseOrderItemFormSet(
                instance=self.object, prefix='items'
            )
        return context
    
    def form_valid(self, form):
        context = self.get_context_data()
        formset = context['formset']
        
        with transaction.atomic():
            self.object = form.save()
            
            if formset.is_valid():
                formset.instance = self.object
                formset.save()
                
                # Update product stock based on quantity changes
                for form in formset:
                    if form.instance.id:  # Existing item
                        old_quantity = PurchaseOrderItem.objects.get(pk=form.instance.id).quantity
                        quantity_diff = form.instance.quantity - old_quantity
                        if quantity_diff != 0:
                            form.instance.product.stock += quantity_diff
                            form.instance.product.save()
                
                messages.success(self.request, 'Purchase order updated successfully.')
                return redirect('purchases:purchase_detail', pk=self.object.pk)
            
        return self.render_to_response(self.get_context_data(form=form))

class PurchaseOrderDetailView(LoginRequiredMixin, DetailView):
    model = PurchaseOrder
    template_name = 'purchases/purchaseorder_detail.html'
    context_object_name = 'purchase_order'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['goods_receipts'] = self.object.goods_receipts.all()
        return context

class PurchaseOrderDeleteView(LoginRequiredMixin, UserPassesTestMixin, DeleteView):
    model = PurchaseOrder
    template_name = 'purchases/purchaseorder_confirm_delete.html'
    success_url = reverse_lazy('purchase_list')
    
    def test_func(self):
        purchase_order = self.get_object()
        return (self.request.user == purchase_order.created_by or 
                self.request.user.is_superuser)
    
    def delete(self, request, *args, **kwargs):
        messages.success(request, 'Purchase order deleted successfully.')
        return super().delete(request, *args, **kwargs)

class GoodsReceiptCreateView(LoginRequiredMixin, CreateView):
    model = GoodsReceipt
    form_class = GoodsReceiptForm
    template_name = 'purchases/goodsreceipt_form.html'
    
    def dispatch(self, request, *args, **kwargs):
        self.purchase_order = get_object_or_404(PurchaseOrder, pk=kwargs['po_pk'])
        return super().dispatch(request, *args, **kwargs)
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['purchase_order'] = self.purchase_order
        
        if self.request.POST:
            context['formset'] = get_goods_receipt_item_formset(
                self.purchase_order,
                data=self.request.POST,
                prefix='items'
            )
        else:
            context['formset'] = get_goods_receipt_item_formset(
                self.purchase_order,
                prefix='items'
            )
            
        return context
    
    def form_valid(self, form):
        context = self.get_context_data()
        formset = context['formset']
        
        with transaction.atomic():
            form.instance.purchase_order = self.purchase_order
            form.instance.received_by = self.request.user
            self.object = form.save()
            
            if formset.is_valid():
                formset.instance = self.object
                formset.save()
                
                # Update purchase order status if all items are received
                self._update_purchase_order_status()
                
                messages.success(self.request, 'Goods receipt created successfully.')
                return redirect('purchases:purchase_detail', pk=self.purchase_order.pk)
            
        return self.render_to_response(self.get_context_data(form=form))
    
    def _update_purchase_order_status(self):
        # Check if all items are fully received
        all_received = all(
            item.quantity <= item.received_quantity 
            for item in self.purchase_order.items.all()
        )
        
        if all_received and self.purchase_order.status != PurchaseOrder.COMPLETED:
            self.purchase_order.status = PurchaseOrder.COMPLETED
            self.purchase_order.save()

class GoodsReceiptDetailView(LoginRequiredMixin, DetailView):
    model = GoodsReceipt
    template_name = 'purchases/goodsreceipt_detail.html'
    context_object_name = 'goods_receipt'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['company_settings'] = getattr(self.request, 'company_settings', None)
        return context

class GoodsReceiptPrintView(LoginRequiredMixin, DetailView):
    model = GoodsReceipt
    template_name = 'purchases/goodsreceipt_print.html'
    context_object_name = 'goods_receipt'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['company_settings'] = getattr(self.request, 'company_settings', None)
        return context
