from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required, user_passes_test
from django.contrib import messages
from django.db.models import Count, Q
from django.http import HttpResponseRedirect
from django.urls import reverse
from .models import Branch
from .forms import BranchForm
from core.models import AuditLog
from accounts.models import User

def is_admin(user):
    """
    Check if user is an admin
    """
    return user.is_authenticated and user.role == 'admin'

def is_system_owner(user):
    """
    Check if user is a system owner
    """
    return user.is_authenticated and hasattr(user, 'is_system_owner') and user.is_system_owner

@login_required
@user_passes_test(lambda u: u.role == 'admin' or u.is_superuser or (hasattr(u, 'is_system_owner') and u.is_system_owner))
def branch_list(request):
    """
    List all branches (admin and system owner only)
    """
    branches = Branch.objects.all().annotate(
        user_count=Count('user', distinct=True),
        product_count=Count('product', distinct=True),
        active_license=Count('licenses', filter=Q(licenses__is_active=True)),
    ).order_by('name')
    
    context = {
        'branches': branches,
    }
    return render(request, 'branches/branch_list.html', context)

@login_required
@user_passes_test(lambda u: u.role == 'admin' or u.is_superuser or (hasattr(u, 'is_system_owner') and u.is_system_owner))
def branch_add(request):
    """
    Add a new branch (admin and system owner only)
    """
    if request.method == 'POST':
        form = BranchForm(request.POST)
        if form.is_valid():
            branch = form.save()
            
            # Log the action
            AuditLog.log_action(
                user=request.user,
                action='create',
                entity_type='branch',
                entity_id=branch.id,
                details=f"Created branch: {branch.name}",
                ip_address=request.META.get('REMOTE_ADDR')
            )
            
            messages.success(request, f'Branch "{branch.name}" created successfully.')
            return redirect('branch_list')
    else:
        form = BranchForm()
    
    context = {
        'form': form,
        'title': 'Add Branch',
    }
    return render(request, 'branches/branch_form.html', context)

@login_required
@user_passes_test(lambda u: u.role == 'admin' or u.is_superuser or (hasattr(u, 'is_system_owner') and u.is_system_owner))
def branch_edit(request, branch_id):
    """
    Edit an existing branch (admin and system owner only)
    """
    branch = get_object_or_404(Branch, id=branch_id)
    
    if request.method == 'POST':
        form = BranchForm(request.POST, instance=branch)
        if form.is_valid():
            branch = form.save()
            
            # Log the action
            AuditLog.log_action(
                user=request.user,
                action='update',
                entity_type='branch',
                entity_id=branch.id,
                details=f"Updated branch: {branch.name}",
                ip_address=request.META.get('REMOTE_ADDR')
            )
            
            messages.success(request, f'Branch "{branch.name}" updated successfully.')
            return redirect('branch_list')
    else:
        form = BranchForm(instance=branch)
    
    context = {
        'form': form,
        'title': 'Edit Branch',
        'branch': branch,
    }
    return render(request, 'branches/branch_form.html', context)

@login_required
@user_passes_test(lambda u: u.role == 'admin' or u.is_superuser or (hasattr(u, 'is_system_owner') and u.is_system_owner))
def branch_delete(request, branch_id):
    """
    Delete a branch (admin and system owner only)
    """
    branch = get_object_or_404(Branch, id=branch_id)
    
    # Get related counts for the confirmation page
    user_count = branch.user_set.count()
    
    # Check if branch has users
    if user_count > 0:
        messages.error(request, f'Cannot delete branch "{branch.name}" because it has {user_count} users assigned to it. Please reassign or delete these users first.')
        return redirect('branch_list')
    
    # Get product count if the model exists
    try:
        from products.models import Product
        product_count = Product.objects.filter(branch=branch).count()
    except (ImportError, ModuleNotFoundError):
        product_count = 0
    
    # Get sales count if the model exists
    try:
        from sales.models import Sale
        sales_count = Sale.objects.filter(branch=branch).count()
    except (ImportError, ModuleNotFoundError):
        sales_count = 0
        
    if request.method == 'POST':
        branch_name = branch.name
        branch.delete()
        
        # Log the action
        AuditLog.log_action(
            user=request.user,
            action='delete',
            entity_type='branch',
            entity_id=branch_id,
            details=f"Deleted branch: {branch_name}",
            ip_address=request.META.get('REMOTE_ADDR')
        )
        
        messages.success(request, f'Branch "{branch_name}" has been deleted successfully.')
        return redirect('branch_list')
    
    context = {
        'branch': branch,
        'user_count': user_count,
        'product_count': product_count,
        'sales_count': sales_count,
    }
    return render(request, 'branches/branch_confirm_delete.html', context)

@login_required
def switch_branch(request, branch_id=None):
    """
    Switch to a different branch (system owner only)
    """
    # Check if user is a system owner or superuser
    if not (request.user.is_superuser or (hasattr(request.user, 'is_system_owner') and request.user.is_system_owner)):
        messages.error(request, 'Permission denied. Only system owners can switch branches.')
        return redirect('dashboard')
        
    if branch_id:
        # Verify the branch exists before setting it
        from .models import Branch
        try:
            branch = Branch.objects.get(id=branch_id)
            # Store the selected branch ID in the session
            request.session['selected_branch_id'] = branch_id
            messages.success(request, f'Switched to branch: {branch.name}')
        except Branch.DoesNotExist:
            messages.error(request, 'Branch not found.')
            if 'selected_branch_id' in request.session:
                del request.session['selected_branch_id']
    else:
        # Clear the selected branch
        if 'selected_branch_id' in request.session:
            del request.session['selected_branch_id']
        messages.success(request, 'Viewing all branches.')
    
    # Redirect back to the referring page or dashboard
    referer = request.META.get('HTTP_REFERER')
    if referer and '/branches/' not in referer and '/accounts/' not in referer:
        return redirect(referer)
    return redirect('dashboard')
