from django.views.generic import ListView, CreateView, UpdateView, DeleteView, DetailView, View, TemplateView
from django.views.decorators.http import require_http_methods
from django.http import JsonResponse
import csv
from django.http import HttpResponse
from django.urls import reverse_lazy
from django.contrib import messages
from django.views.decorators.csrf import csrf_exempt
from django.utils.decorators import method_decorator
from django.contrib.auth.mixins import LoginRequiredMixin, UserPassesTestMixin
from django.shortcuts import redirect
from django.utils.translation import gettext_lazy as _
from django.db import models
from django.db.models import Q, Sum

from .models import Expense, ExpenseCategory
from .forms import ExpenseForm, ExpenseCategoryForm

class ExpenseListView(LoginRequiredMixin, ListView):
    model = Expense
    template_name = 'expenses/expense_list.html'
    context_object_name = 'expenses'
    paginate_by = 20
    
    def get_queryset(self):
        queryset = Expense.objects.select_related('category', 'created_by')
        
        # Filter by status if provided
        status = self.request.GET.get('status')
        if status:
            queryset = queryset.filter(status=status)
        
        # Filter by date range if provided
        date_from = self.request.GET.get('date_from')
        date_to = self.request.GET.get('date_to')
        
        if date_from:
            queryset = queryset.filter(expense_date__gte=date_from)
        if date_to:
            queryset = queryset.filter(expense_date__lte=date_to)
        
        # Filter by search query if provided
        search = self.request.GET.get('q')
        if search:
            queryset = queryset.filter(
                Q(description__icontains=search) |
                Q(reference__icontains=search) |
                Q(notes__icontains=search)
            )
        
        # For non-superusers, only show their own expenses
        if not self.request.user.is_superuser:
            queryset = queryset.filter(created_by=self.request.user)
        
        return queryset.order_by('-expense_date', '-created_at')
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        queryset = self.get_queryset()
        
        # Calculate total amount from the filtered queryset
        context['total_amount'] = queryset.aggregate(total=models.Sum('amount'))['total'] or 0
        
        # Add other context data
        context['status_choices'] = dict(Expense.STATUS_CHOICES)
        context['current_status'] = self.request.GET.get('status', '')
        context['date_from'] = self.request.GET.get('date_from', '')
        context['date_to'] = self.request.GET.get('date_to', '')
        context['search_query'] = self.request.GET.get('q', '')
        
        # Add categories for filter dropdown
        context['categories'] = ExpenseCategory.objects.filter(is_active=True)
        
        return context

class ExpenseCreateView(LoginRequiredMixin, CreateView):
    model = Expense
    form_class = ExpenseForm
    template_name = 'expenses/expense_form.html'
    
    def get_form_kwargs(self):
        kwargs = super().get_form_kwargs()
        kwargs['request'] = self.request
        return kwargs
    
    def form_valid(self, form):
        # Handle new category creation
        new_category_name = form.cleaned_data.pop('new_category', None)
        if new_category_name:
            category, created = ExpenseCategory.objects.get_or_create(
                name=new_category_name.strip(),
                defaults={'is_active': True}
            )
            form.instance.category = category
        
        # Set the created_by user
        form.instance.created_by = self.request.user
        
        # Set the branch if user has one
        if hasattr(self.request.user, 'employee') and hasattr(self.request.user.employee, 'branch'):
            form.instance.branch = self.request.user.employee.branch
        
        response = super().form_valid(form)
        messages.success(
            self.request,
            _('Expense has been created successfully.')
        )
        return response
    
    def get_success_url(self):
        return reverse_lazy('expenses:expense_list')

class ExpenseUpdateView(LoginRequiredMixin, UserPassesTestMixin, UpdateView):
    model = Expense
    form_class = ExpenseForm
    template_name = 'expenses/expense_form.html'
    
    def test_func(self):
        expense = self.get_object()
        return self.request.user.is_superuser or expense.created_by == self.request.user
    
    def get_form_kwargs(self):
        kwargs = super().get_form_kwargs()
        kwargs['request'] = self.request
        return kwargs
    
    def form_valid(self, form):
        response = super().form_valid(form)
        messages.success(
            self.request,
            _('Expense has been updated successfully.')
        )
        return response
    
    def get_success_url(self):
        return reverse_lazy('expenses:expense_list')

class ExpenseDeleteView(LoginRequiredMixin, UserPassesTestMixin, DeleteView):
    model = Expense
    template_name = 'expenses/expense_confirm_delete.html'
    success_url = reverse_lazy('expenses:expense_list')
    
    def test_func(self):
        expense = self.get_object()
        return self.request.user.is_superuser or expense.created_by == self.request.user
    
    def delete(self, request, *args, **kwargs):
        response = super().delete(request, *args, **kwargs)
        messages.success(
            request,
            _('Expense has been deleted successfully.')
        )
        return response

class ExpenseDetailView(LoginRequiredMixin, UserPassesTestMixin, DetailView):
    model = Expense
    template_name = 'expenses/expense_detail.html'
    context_object_name = 'expense'
    
    def test_func(self):
        expense = self.get_object()
        return (self.request.user.is_superuser or 
                expense.created_by == self.request.user or
                self.request.user.has_perm('expenses.view_all_expenses'))

# Category Views
class ExpenseCategoryListView(LoginRequiredMixin, UserPassesTestMixin, ListView):
    model = ExpenseCategory
    template_name = 'expenses/category_list.html'
    context_object_name = 'categories'
    
    def test_func(self):
        return self.request.user.is_superuser or self.request.user.has_perm('expenses.manage_categories')
    
    def get_queryset(self):
        return ExpenseCategory.objects.all().order_by('name')

class ExpenseCategoryCreateView(LoginRequiredMixin, UserPassesTestMixin, CreateView):
    model = ExpenseCategory
    form_class = ExpenseCategoryForm
    template_name = 'expenses/category_form.html'
    success_url = reverse_lazy('expenses:category_list')
    
    def test_func(self):
        return self.request.user.is_superuser or self.request.user.has_perm('expenses.manage_categories')
    
    def form_valid(self, form):
        response = super().form_valid(form)
        messages.success(self.request, _('Category has been created successfully.'))
        return response

@method_decorator(csrf_exempt, name='dispatch')
class QuickAddCategoryView(LoginRequiredMixin, View):
    """View to handle quick category addition via AJAX"""
    
    def post(self, request, *args, **kwargs):
        name = request.POST.get('name', '').strip()
        if not name:
            return JsonResponse({'success': False, 'error': 'Category name is required'})
            
        # Check if category already exists (case-insensitive)
        category, created = ExpenseCategory.objects.get_or_create(
            name__iexact=name,
            defaults={'name': name, 'is_active': True}
        )
        
        if not created:
            # Category already exists, just return it
            return JsonResponse({
                'success': True,
                'message': 'Category already exists',
                'category': {
                    'id': category.id,
                    'name': category.name
                }
            })
            
        return JsonResponse({
            'success': True,
            'message': 'Category created successfully',
            'category': {
                'id': category.id,
                'name': category.name
            }
        })

class ExpenseCategoryUpdateView(LoginRequiredMixin, UserPassesTestMixin, UpdateView):
    model = ExpenseCategory
    form_class = ExpenseCategoryForm
    template_name = 'expenses/category_form.html'
    success_url = reverse_lazy('expenses:category_list')
    
    def test_func(self):
        return self.request.user.is_superuser or self.request.user.has_perm('expenses.manage_categories')
    
    def form_valid(self, form):
        response = super().form_valid(form)
        messages.success(self.request, _('Category has been updated successfully.'))
        return response

class ExpenseCategoryDeleteView(LoginRequiredMixin, UserPassesTestMixin, DeleteView):
    model = ExpenseCategory
    template_name = 'expenses/category_confirm_delete.html'
    success_url = reverse_lazy('expenses:category_list')
    
    def test_func(self):
        return self.request.user.is_superuser or self.request.user.has_perm('expenses.manage_categories')
    
    def delete(self, request, *args, **kwargs):
        response = super().delete(request, *args, **kwargs)
        messages.success(request, _('Category has been deleted successfully.'))
        return response


class SingleExpenseExportView(LoginRequiredMixin, View):
    """View to export a single expense to CSV"""
    
    def get(self, request, *args, **kwargs):
        try:
            expense = Expense.objects.get(pk=self.kwargs['pk'])
            
            # Create the HttpResponse object with CSV header
            response = HttpResponse(content_type='text/csv')
            response['Content-Disposition'] = f'attachment; filename="expense_{expense.reference}.csv"'
            
            # Create the CSV writer
            writer = csv.writer(response)
            
            # Write headers
            writer.writerow([
                'Reference', 'Date', 'Category', 'Amount', 'Tax Rate (%)', 
                'Tax Amount', 'Total Amount', 'Payment Method', 'Status', 
                'Description', 'Created By', 'Created At', 'Updated At'
            ])
            
            # Write expense data
            writer.writerow([
                expense.reference,
                expense.expense_date.strftime('%Y-%m-%d') if expense.expense_date else '',
                str(expense.category) if expense.category else '',
                str(expense.amount),
                str(expense.tax_rate) if expense.tax_rate is not None else '0',
                str(expense.tax_amount) if expense.tax_amount is not None else '0',
                str(expense.total_amount),
                expense.get_payment_method_display(),
                expense.get_status_display(),
                expense.description or '',
                str(expense.created_by) if expense.created_by else '',
                expense.created_at.strftime('%Y-%m-%d %H:%M:%S') if expense.created_at else '',
                expense.updated_at.strftime('%Y-%m-%d %H:%M:%S') if expense.updated_at else ''
            ])
            
            return response
            
        except Expense.DoesNotExist:
            messages.error(request, 'Expense not found.')
            return redirect('expenses:expense_list')


class ExpensePrintView(LoginRequiredMixin, DetailView):
    model = Expense
    template_name = 'expenses/expense_print.html'
    context_object_name = 'expense'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['company_settings'] = get_company_settings()
        return context

class CashierExpenseCreateView(LoginRequiredMixin, CreateView):
    """View for cashiers to quickly add expenses from dashboard"""
    model = Expense
    form_class = ExpenseForm
    template_name = 'expenses/cashier_expense_form.html'
    
    def get_form_kwargs(self):
        kwargs = super().get_form_kwargs()
        kwargs['request'] = self.request
        return kwargs
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['categories'] = ExpenseCategory.objects.filter(is_active=True)
        return context
    
    def form_valid(self, form):
        # Handle new category creation
        new_category_name = form.cleaned_data.pop('new_category', None)
        if new_category_name:
            category, created = ExpenseCategory.objects.get_or_create(
                name=new_category_name.strip(),
                defaults={'is_active': True}
            )
            form.instance.category = category
        
        # Set the created_by user and status
        form.instance.created_by = self.request.user
        form.instance.status = 'approved'  # Auto-approve cashier expenses
        
        # Set the branch if user has one
        if hasattr(self.request.user, 'employee') and hasattr(self.request.user.employee, 'branch'):
            form.instance.branch = self.request.user.employee.branch
        
        messages.success(self.request, 'Expense added successfully.')
        return super().form_valid(form)
    
    def get_success_url(self):
        return reverse_lazy('dashboard')  # Redirect back to dashboard


class ExpenseExportView(LoginRequiredMixin, View):
    def get(self, request, *args, **kwargs):
        # Get the filtered queryset using the same logic as ExpenseListView
        queryset = Expense.objects.select_related('category', 'created_by')
        
        # Apply filters
        status = request.GET.get('status')
        date_from = request.GET.get('date_from')
        date_to = request.GET.get('date_to')
        search = request.GET.get('q')
        
        if status:
            queryset = queryset.filter(status=status)
        if date_from:
            queryset = queryset.filter(expense_date__gte=date_from)
        if date_to:
            queryset = queryset.filter(expense_date__lte=date_to)
        if search:
            queryset = queryset.filter(
                Q(description__icontains=search) |
                Q(reference__icontains=search) |
                Q(notes__icontains=search)
            )
        if not request.user.is_superuser:
            queryset = queryset.filter(created_by=request.user)
        
        # Create the HttpResponse object with the appropriate CSV header.
        response = HttpResponse(
            content_type='text/csv',
            headers={'Content-Disposition': 'attachment; filename="expenses_export.csv"'},
        )
        
        # Create the CSV writer
        writer = csv.writer(response)
        
        # Write the header row
        writer.writerow([
            'Reference', 'Date', 'Category', 'Description', 'Amount', 
            'Status', 'Payment Method', 'Created By', 'Created At', 'Notes'
        ])
        
        # Write data rows
        for expense in queryset.order_by('-expense_date', '-created_at'):
            writer.writerow([
                expense.reference,
                expense.expense_date.strftime('%Y-%m-%d') if expense.expense_date else '',
                str(expense.category) if expense.category else '',
                expense.description or '',
                str(expense.amount) if expense.amount is not None else '0.00',
                expense.get_status_display(),
                expense.get_payment_method_display(),
                expense.created_by.get_full_name() or expense.created_by.username,
                expense.created_at.strftime('%Y-%m-%d %H:%M:%S') if expense.created_at else '',
                expense.notes or ''
            ])
        
        return response
