from django.test import TestCase
from django.urls import reverse
from django.contrib.auth import get_user_model
from decimal import Decimal
from .models import Customer
from branches.models import Branch
from sales.models import Sale

User = get_user_model()

class CustomerModelTest(TestCase):
    def setUp(self):
        self.branch = Branch.objects.create(
            name='Test Branch',
            address='123 Test St'
        )
        
        self.customer = Customer.objects.create(
            first_name='John',
            last_name='Doe',
            email='john@example.com',
            phone='123-456-7890',
            address='123 Main St',
            city='Anytown',
            state='ST',
            postal_code='12345',
            branch=self.branch
        )
    
    def test_customer_creation(self):
        self.assertEqual(Customer.objects.count(), 1)
        self.assertEqual(self.customer.full_name, 'John Doe')
    
    def test_customer_properties(self):
        self.assertEqual(self.customer.full_address, '123 Main St, Anytown, ST, 12345')
        self.assertEqual(self.customer.total_purchases, 0)
        self.assertEqual(self.customer.total_spent, 0)
    
    def test_purchases_and_spending(self):
        # Create a user for the sale
        user = User.objects.create_user(
            email='testuser@example.com',
            password='testpass123',
            first_name='Test',
            last_name='User',
            role='cashier',
            branch=self.branch
        )
        
        # Create a sale for the customer
        sale1 = Sale.objects.create(
            customer=self.customer,
            cashier=user,
            branch=self.branch,
            total_amount=Decimal('100.00'),
            payment_status='paid'
        )
        
        sale2 = Sale.objects.create(
            customer=self.customer,
            cashier=user,
            branch=self.branch,
            total_amount=Decimal('50.00'),
            payment_status='pending'
        )
        
        # Test purchase count and total spent
        self.assertEqual(self.customer.total_purchases, 2)
        self.assertEqual(self.customer.total_spent, Decimal('100.00'))  # Only paid sales

class CustomerViewTest(TestCase):
    def setUp(self):
        self.branch = Branch.objects.create(
            name='Test Branch',
            address='123 Test St'
        )
        
        self.user = User.objects.create_user(
            email='testuser@example.com',
            password='testpassword',
            first_name='Test',
            last_name='User',
            role='admin',
            branch=self.branch
        )
        
        self.client.login(username='testuser@example.com', password='testpassword')
        
        self.customer = Customer.objects.create(
            first_name='John',
            last_name='Doe',
            email='john@example.com',
            phone='123-456-7890',
            branch=self.branch
        )
    
    def test_customer_list_view(self):
        response = self.client.get(reverse('customer_list'))
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'customers/customer_list.html')
        self.assertContains(response, 'John Doe')
    
    def test_customer_create_view(self):
        response = self.client.post(reverse('customer_add'), {
            'first_name': 'Jane',
            'last_name': 'Smith',
            'email': 'jane@example.com',
            'phone': '555-555-5555',
            'branch': self.branch.id
        })
        self.assertRedirects(response, reverse('customer_list'))
        self.assertEqual(Customer.objects.count(), 2)
    
    def test_customer_detail_view(self):
        response = self.client.get(reverse('customer_detail', args=[self.customer.id]))
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'customers/customer_detail.html')
        self.assertContains(response, 'John Doe')
