from .models import Notification, CompanySettings
from branches.models import Branch

def company_settings_processor(request):
    """
    Context processor for company settings
    """
    try:
        # If user is authenticated and has a branch, get branch-specific settings
        if request.user.is_authenticated and hasattr(request, 'branch') and request.branch:
            settings = CompanySettings.objects.filter(branch=request.branch).first()
            if not settings:
                # Fallback to global settings if branch-specific not found
                settings = CompanySettings.objects.filter(branch__isnull=True).first()
        else:
            # Get global settings
            settings = CompanySettings.objects.filter(branch__isnull=True).first()
    except:
        settings = None
    
    return {'company_settings': settings}

def current_branch_processor(request):
    """
    Context processor for current branch information
    """
    if not request.user.is_authenticated:
        return {'current_branch': None, 'all_branches': []}
    
    current_branch = getattr(request, 'branch', None)
    if not current_branch and hasattr(request.user, 'branch'):
        current_branch = request.user.branch
    
    # For system owners and superusers, provide a list of all branches
    if hasattr(request.user, 'is_system_owner') and request.user.is_system_owner or request.user.is_superuser:
        all_branches = Branch.objects.all().order_by('name')
    else:
        all_branches = []
    
    return {
        'current_branch': current_branch,
        'all_branches': all_branches
    }

def sidebar_items(request):
    """
    Context processor for sidebar menu items based on user role
    """
    if not request.user.is_authenticated:
        return {}
    
    # Base menu items for all roles
    menu_items = [
        {'name': 'Dashboard', 'url': 'dashboard', 'icon': 'fas fa-tachometer-alt'},
        {'name': 'Make Sale', 'url': 'new_sale', 'icon': 'fas fa-cash-register'}
    ]
    
    # System Owner menu items
    if hasattr(request.user, 'is_system_owner') and request.user.is_system_owner:
        system_owner_menu = []
        
        # Add menu items safely
        try:
            from django.urls import reverse, NoReverseMatch
            
            # Core system owner menu items
            safe_urls = [
                {'name': 'Dashboard', 'url': 'dashboard', 'icon': 'fas fa-tachometer-alt'},
                {'name': 'Branches', 'url': 'branch_list', 'icon': 'fas fa-store'},
                {'name': 'Users', 'url': 'user_list', 'icon': 'fas fa-user-cog'},
                {'name': 'Settings', 'url': 'company_settings', 'icon': 'fas fa-cogs'}
            ]
            
            # Add items only if their URLs exist
            for item in safe_urls:
                try:
                    reverse(item['url'])
                    system_owner_menu.append(item)
                except NoReverseMatch:
                    pass
            
            # Try to add license management
            try:
                reverse('license_list')
                system_owner_menu.append({'name': 'Licenses', 'url': 'license_list', 'icon': 'fas fa-key'})
            except NoReverseMatch:
                pass
                
            # Try to add business creation
            try:
                reverse('admin_create_business')
                system_owner_menu.append({'name': 'Businesses', 'url': 'admin_create_business', 'icon': 'fas fa-building'})
            except NoReverseMatch:
                pass
                
            # Try to add audit logs
            try:
                reverse('audit_logs')
                system_owner_menu.append({'name': 'Audit Logs', 'url': 'audit_logs', 'icon': 'fas fa-history'})
            except NoReverseMatch:
                pass
        except ImportError:
            # Fallback if there's an issue with imports
            system_owner_menu = [
                {'name': 'Dashboard', 'url': 'dashboard', 'icon': 'fas fa-tachometer-alt'},
                {'name': 'Branches', 'url': 'branch_list', 'icon': 'fas fa-store'},
                {'name': 'Users', 'url': 'user_list', 'icon': 'fas fa-user-cog'}
            ]
            
        menu_items = system_owner_menu
    
    # Admin menu items
    elif request.user.role == 'admin' or request.user.is_superuser:
        admin_menu = []
        
        # All menu items in a single list
        menu_items_list = [
            {'name': 'All Sales', 'url': 'sale_list', 'icon': 'fas fa-shopping-cart'},
            {'name': 'Products', 'url': 'product_list', 'icon': 'fas fa-box'},
            {'name': 'Categories', 'url': 'category_list', 'icon': 'fas fa-tags'},
            {'name': 'Customers', 'url': 'customer_list', 'icon': 'fas fa-users'},
            {'name': 'Purchases', 'url': 'purchases:purchase_list', 'icon': 'fas fa-shopping-bag'},
            {'name': 'Quotations', 'url': 'quotations:quotation_list', 'icon': 'fas fa-file-invoice'},
            {'name': 'Expenses', 'url': 'expenses:expense_list', 'icon': 'fas fa-money-bill-wave'},
            {'name': 'Suppliers', 'url': 'suppliers:supplier_list', 'icon': 'fas fa-truck'},
            {'name': 'Branches', 'url': 'branch_list', 'icon': 'fas fa-store'},
            {'name': 'Users', 'url': 'user_list', 'icon': 'fas fa-user-cog'},
            {'name': 'Reports', 'url': 'reports', 'icon': 'fas fa-chart-bar'},
            {'name': 'Settings', 'url': 'company_settings', 'icon': 'fas fa-cogs'}
        ]
        
        # Add menu items safely
        try:
            from django.urls import reverse
            
            for item in menu_items_list:
                try:
                    reverse(item['url'])
                    admin_menu.append(item)
                except Exception as e:
                    print(f"Skipping {item['name']} - URL not found: {e}")
                    continue
                    
        except Exception as e:
            print(f"Error adding menu items: {e}")
        except:
            # If any url is not found, just skip it
            pass
            
        menu_items.extend(admin_menu)
    
    # Manager menu items
    elif request.user.role == 'manager':
        manager_menu = [
            {'name': 'Branches', 'url': 'branch_list', 'icon': 'fas fa-store'},
            {'name': 'Settings', 'url': 'company_settings', 'icon': 'fas fa-cogs'}
        ]
        
        # Add feature menu items only if they have corresponding url patterns
        try:
            from django.urls import reverse
            if reverse('sale_list'):
                manager_menu.insert(0, {'name': 'All Sales', 'url': 'sale_list', 'icon': 'fas fa-shopping-cart'})
            if reverse('product_list'):
                manager_menu.insert(1, {'name': 'Products', 'url': 'product_list', 'icon': 'fas fa-box'})
            # Add Categories menu item for manager
            try:
                if reverse('category_list'):
                    manager_menu.insert(2, {'name': 'Categories', 'url': 'category_list', 'icon': 'fas fa-tags'})
            except:
                pass
            if reverse('customer_list'):
                manager_menu.insert(3, {'name': 'Customers', 'url': 'customer_list', 'icon': 'fas fa-users'})
            # Add new menu items for managers
            try:
                if reverse('purchase_list'):
                    manager_menu.insert(4, {'name': 'Purchases', 'url': 'purchase_list', 'icon': 'fas fa-shopping-bag'})
                if reverse('supplier_list'):
                    manager_menu.insert(5, {'name': 'Suppliers', 'url': 'supplier_list', 'icon': 'fas fa-truck'})
            except:
                pass
            if reverse('reports'):
                manager_menu.append({'name': 'Reports', 'url': 'reports', 'icon': 'fas fa-chart-bar'})
        except:
            # If any url is not found, just skip it
            pass
            
        menu_items.extend(manager_menu)
    
    # Cashier menu items
    elif request.user.role == 'cashier':
        cashier_menu = []
        
        # Add feature menu items only if they have corresponding url patterns
        try:
            from django.urls import reverse
            if reverse('sale_list'):
                cashier_menu.append({'name': 'All Sales', 'url': 'sale_list', 'icon': 'fas fa-shopping-cart'})
            if reverse('product_list'):
                cashier_menu.append({'name': 'Products', 'url': 'product_list', 'icon': 'fas fa-box'})
            # Categories are hidden from cashiers
            # Categories are only accessible to admins and managers
            if reverse('customer_list'):
                cashier_menu.append({'name': 'Customers', 'url': 'customer_list', 'icon': 'fas fa-users'})
            # Add expenses for cashiers
            try:
                if reverse('expenses:expense_list'):
                    cashier_menu.append({
                        'name': 'Expenses', 
                        'url': 'expenses:expense_list', 
                        'icon': 'fas fa-money-bill-wave'
                    })
                
                # Add limited access to purchases for cashiers (view only)
                if reverse('purchase_list'):
                    cashier_menu.append({
                        'name': 'View Purchases', 
                        'url': 'purchase_list', 
                        'icon': 'fas fa-shopping-bag'
                    })
                    
            except Exception as e:
                print(f"Error adding menu items: {str(e)}")
        except:
            # If any url is not found, just skip it
            pass
            
        menu_items.extend(cashier_menu)
    
    return {'sidebar_items': menu_items}

def notification_count(request):
    """
    Context processor for notification count
    Shows all notifications without any limit
    """
    if not request.user.is_authenticated:
        return {'notification_count': 0, 'notifications': []}
    
    # Get all unread notifications
    if hasattr(request.user, 'is_system_owner') and request.user.is_system_owner:
        # System owners see all notifications
        notifications = Notification.objects.filter(
            is_read=False
        ).order_by('-created_at')
    elif request.user.role != 'admin':
        # Non-admin users see only their branch notifications
        notifications = Notification.objects.filter(
            branch=request.user.branch, 
            is_read=False
        ).order_by('-created_at')
    else:
        # Regular admins see their branch notifications or all if no branch
        if hasattr(request.user, 'branch') and request.user.branch:
            notifications = Notification.objects.filter(
                branch=request.user.branch,
                is_read=False
            ).order_by('-created_at')
        else:
            notifications = Notification.objects.filter(
                is_read=False
            ).order_by('-created_at')
    
    # Return all notifications without any filtering by type
    return {
        'notification_count': notifications.count(),
        'notifications': notifications
    }
