from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required, user_passes_test
from django.contrib import messages
from django.utils import timezone
from datetime import timedelta, date
from decimal import Decimal
import json
from .models import CompanySettings, Notification, AuditLog
from .forms import CompanySettingsForm
from accounts.models import User
from licenses.views import check_license_required

# Custom JSON encoder to handle date and Decimal objects
class CustomJSONEncoder(json.JSONEncoder):
    def default(self, o):
        if isinstance(o, date):
            return o.isoformat()
        if isinstance(o, Decimal):
            return float(o)
        return super().default(o)

# Temporary placeholders until we implement the full modules
try:
    from sales.models import Sale, SaleItem
    from products.models import Product
    SALES_MODULE_ENABLED = True
except ImportError:
    SALES_MODULE_ENABLED = False
    # Create placeholder classes for type checking
    class Sale:
        @staticmethod
        def objects():
            return None
            
    class SaleItem:
        @staticmethod
        def objects():
            return None
            
    class Product:
        @staticmethod
        def objects():
            return None

def is_admin_or_manager_or_system_owner(user):
    """
    Check if user is an admin, manager, or system owner
    """
    return user.is_authenticated and (user.role in ['admin', 'manager', 'system_owner'] or user.is_superuser)

@login_required
@check_license_required
def dashboard(request):
    """
    Main dashboard view showing metrics based on user role
    """
    # Date ranges for filtering
    today = timezone.now().date()
    start_of_week = today - timedelta(days=today.weekday())
    start_of_month = today.replace(day=1)
    
    # Check if sales modules are available
    if SALES_MODULE_ENABLED:
        from django.db.models import Sum, Count, F
        from django.db.models.functions import TruncDate
        
        # Base queryset filtered by user's branch if not admin
        if request.user.role != 'admin':
            sales_queryset = Sale.objects.filter(branch=request.user.branch)
            products_queryset = Product.objects.filter(branch=request.user.branch)
        else:
            sales_queryset = Sale.objects.all()
            products_queryset = Product.objects.all()
        
        # Today's sales
        today_sales = sales_queryset.filter(created_at__date=today)
        today_sales_count = today_sales.count()
        today_sales_total = today_sales.aggregate(total=Sum('total_amount'))['total'] or 0
        
        # Weekly sales
        week_sales = sales_queryset.filter(created_at__date__gte=start_of_week)
        week_sales_count = week_sales.count()
        week_sales_total = week_sales.aggregate(total=Sum('total_amount'))['total'] or 0
        
        # Monthly sales
        month_sales = sales_queryset.filter(created_at__date__gte=start_of_month)
        month_sales_count = month_sales.count()
        month_sales_total = month_sales.aggregate(total=Sum('total_amount'))['total'] or 0
        
        # Daily sales chart data for the last 7 days
        last_7_days = today - timedelta(days=6)
        daily_sales = sales_queryset.filter(
            created_at__date__gte=last_7_days
        ).annotate(
            date=TruncDate('created_at')
        ).values('date').annotate(
            total=Sum('total_amount')
        ).order_by('date')
        
        # Top selling products
        top_products = SaleItem.objects.filter(
            sale__in=sales_queryset
        ).values(
            'product__name'
        ).annotate(
            total_sold=Sum('quantity')
        ).order_by('-total_sold')[:5]
        
        # Low stock products
        low_stock_products = products_queryset.filter(
            quantity__lt=F('low_stock_threshold')
        )[:5]
        
        # Cashier-specific metrics
        if request.user.role == 'cashier':
            user_sales = sales_queryset.filter(cashier=request.user)
            user_sales_count = user_sales.count()
            user_sales_total = user_sales.aggregate(total=Sum('total_amount'))['total'] or 0
        else:
            user_sales_count = 0
            user_sales_total = 0
            
        # Recent sales
        recent_sales = sales_queryset.order_by('-created_at')[:5]
        
    else:
        # Sales modules not available yet, use empty data
        today_sales_count = 0
        today_sales_total = 0
        week_sales_count = 0
        week_sales_total = 0
        month_sales_count = 0
        month_sales_total = 0
        user_sales_count = 0
        user_sales_total = 0
        
        # Empty collections for template
        daily_sales = []
        top_products = []
        low_stock_products = []
        recent_sales = []
    
    # Prepare context
    context = {
        'today_sales_count': today_sales_count,
        'today_sales_total': today_sales_total,
        'week_sales_count': week_sales_count,
        'week_sales_total': week_sales_total,
        'month_sales_count': month_sales_count,
        'month_sales_total': month_sales_total,
        'daily_sales': json.dumps(list(daily_sales) if hasattr(daily_sales, '__iter__') else [], cls=CustomJSONEncoder),
        'top_products': json.dumps(list(top_products) if hasattr(top_products, '__iter__') else [], cls=CustomJSONEncoder),
        'low_stock_products': low_stock_products,
        'user_sales_count': user_sales_count,
        'user_sales_total': user_sales_total,
        'recent_sales': recent_sales,
        'modules_enabled': {
            'sales': SALES_MODULE_ENABLED
        }
    }
    
    return render(request, 'dashboard/index.html', context)

@login_required
@user_passes_test(is_admin_or_manager_or_system_owner)
def company_settings(request):
    """
    View for editing company settings
    """
    settings = CompanySettings.get_settings()
    
    if request.method == 'POST':
        form = CompanySettingsForm(request.POST, request.FILES, instance=settings)
        if form.is_valid():
            form.save()
            messages.success(request, 'Company settings updated successfully.')
            return redirect('company_settings')
    else:
        form = CompanySettingsForm(instance=settings)
    
    context = {
        'form': form,
        'settings': settings,
    }
    return render(request, 'settings/company_settings.html', context)

@login_required
def notifications(request):
    """
    View for displaying notifications
    """
    # Filter notifications by branch if not admin
    if request.user.role != 'admin':
        notifications = Notification.objects.filter(branch=request.user.branch)
    else:
        notifications = Notification.objects.all()
    
    context = {
        'notifications': notifications,
    }
    return render(request, 'notifications/notification_list.html', context)

@login_required
def mark_notification_read(request, notification_id):
    """
    Mark a notification as read
    """
    notification = get_object_or_404(Notification, id=notification_id)
    
    # Check if user has access to this notification
    if request.user.role != 'admin' and notification.branch != request.user.branch:
        messages.error(request, 'You do not have permission to access this notification.')
        return redirect('notifications')
    
    notification.is_read = True
    notification.save()
    
    next_url = request.GET.get('next', 'notifications')
    return redirect(next_url)

@login_required
def mark_all_notifications_read(request):
    """
    Mark all notifications as read
    """
    # Filter notifications by branch if not admin
    if request.user.role != 'admin':
        Notification.objects.filter(branch=request.user.branch, is_read=False).update(is_read=True)
    else:
        Notification.objects.filter(is_read=False).update(is_read=True)
    
    messages.success(request, 'All notifications marked as read.')
    return redirect('notifications')

@login_required
@user_passes_test(lambda u: u.role == 'admin' or u.role == 'system_owner' or u.is_superuser)
def audit_logs(request):
    """
    View for displaying audit logs (admin only)
    """
    logs = AuditLog.objects.all().select_related('user').order_by('-timestamp')
    
    context = {
        'logs': logs,
    }
    return render(request, 'audit_logs/audit_log_list.html', context)
