from django.db import models
from django.urls import reverse
from django.utils.text import slugify
from django.core.validators import MinValueValidator, MaxValueValidator
from django.core.exceptions import ValidationError
from django.utils import timezone
from django.conf import settings

from branches.models import Branch

class SupplierCategory(models.Model):
    """Categories for organizing suppliers (e.g., Wholesaler, Manufacturer, etc.)"""
    name = models.CharField(max_length=100, unique=True)
    description = models.TextField(blank=True)
    is_active = models.BooleanField(default=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name_plural = 'Supplier Categories'
        ordering = ['name']
    
    def __str__(self):
        return self.name

class Supplier(models.Model):
    """Main supplier model"""
    SUPPLIER_TYPE_CHOICES = [
        ('company', 'Company'),
        ('individual', 'Individual'),
        ('government', 'Government'),
        ('other', 'Other'),
    ]
    
    # Payment terms in days (numeric value)
    
    # Basic Information
    name = models.CharField(max_length=200)
    slug = models.SlugField(max_length=200, unique=True, blank=True)
    supplier_type = models.CharField(max_length=20, choices=SUPPLIER_TYPE_CHOICES, default='company')
    company_name = models.CharField(max_length=200, blank=True)
    tax_identification_number = models.CharField('Tax ID', max_length=50, blank=True)
    registration_number = models.CharField('Registration #', max_length=50, blank=True)
    vat_number = models.CharField('VAT Number', max_length=50, blank=True)
    
    # Contact Information
    email = models.EmailField('Email Address', blank=True)
    phone = models.CharField(max_length=20, blank=True)
    mobile = models.CharField(max_length=20, blank=True)
    website = models.URLField('Website', blank=True)
    
    # Address
    address_line1 = models.CharField('Address Line 1', max_length=255, blank=True)
    address_line2 = models.CharField('Address Line 2', max_length=255, blank=True)
    city = models.CharField(max_length=100, blank=True)
    state = models.CharField(max_length=100, blank=True)
    postal_code = models.CharField('Postal Code', max_length=20, blank=True)
    country = models.CharField(max_length=100, default='Kenya')
    
    # Business Information
    category = models.ForeignKey(SupplierCategory, on_delete=models.SET_NULL, null=True, blank=True, related_name='suppliers')
    currency = models.CharField(max_length=3, default='KES', help_text='Default currency for transactions with this supplier')
    payment_terms = models.PositiveIntegerField(
        default=30,
        help_text='Payment terms in days (e.g., 30 for Net 30)',
        validators=[MinValueValidator(0)]
    )
    credit_limit = models.DecimalField(max_digits=12, decimal_places=2, default=0, help_text='Credit limit for this supplier')
    
    # Status & Relationships
    is_active = models.BooleanField(default=True)
    preferred = models.BooleanField(default=False, help_text='Mark as preferred supplier')
    branches = models.ManyToManyField(Branch, blank=True, related_name='suppliers')
    
    # Financial Information
    account_balance = models.DecimalField(max_digits=12, decimal_places=2, default=0, help_text='Current account balance')
    total_purchases = models.DecimalField(max_digits=12, decimal_places=2, default=0, help_text='Total purchases from this supplier')
    total_payments = models.DecimalField(max_digits=12, decimal_places=2, default=0, help_text='Total payments made to this supplier')
    
    # Audit Fields
    notes = models.TextField(blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    created_by = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.SET_NULL, null=True, related_name='created_suppliers')
    
    class Meta:
        ordering = ['name']
        verbose_name_plural = 'Suppliers'
        
    def __str__(self):
        return self.name
    
    def save(self, *args, **kwargs):
        if not self.slug:
            self.slug = slugify(self.name)
        super().save(*args, **kwargs)
    
    def get_absolute_url(self):
        return reverse('supplier_detail', kwargs={'slug': self.slug})
    
    @property
    def balance(self):
        """Calculate the current balance (total purchases - total payments)"""
        return self.total_purchases - self.total_payments
    
    @property
    def primary_contact(self):
        """Get the primary contact person for this supplier"""
        return self.contacts.filter(is_primary=True).first()
    
    def update_balance(self, amount, transaction_type):
        """Update the supplier's balance based on transaction type"""
        if transaction_type == 'purchase':
            self.total_purchases += amount
        elif transaction_type == 'payment':
            self.total_payments += amount
        self.save()

class SupplierContact(models.Model):
    """Contact persons for suppliers"""
    CONTACT_TYPES = [
        ('primary', 'Primary Contact'),
        ('billing', 'Billing Contact'),
        ('technical', 'Technical Contact'),
        ('other', 'Other'),
    ]
    
    supplier = models.ForeignKey(Supplier, on_delete=models.CASCADE, related_name='contacts')
    first_name = models.CharField(max_length=100)
    last_name = models.CharField(max_length=100)
    position = models.CharField(max_length=100, blank=True)
    email = models.EmailField(blank=True)
    phone = models.CharField(max_length=20, blank=True)
    mobile = models.CharField(max_length=20, blank=True)
    is_primary = models.BooleanField(default=False, help_text='Primary contact for the supplier')
    contact_type = models.CharField(max_length=20, choices=CONTACT_TYPES, default='primary')
    notes = models.TextField(blank=True)
    
    class Meta:
        ordering = ['-is_primary', 'last_name', 'first_name']
        verbose_name = 'Supplier Contact'
        verbose_name_plural = 'Supplier Contacts'
    
    def __str__(self):
        return f"{self.first_name} {self.last_name}"
    
    def save(self, *args, **kwargs):
        # Ensure only one primary contact per supplier
        if self.is_primary:
            SupplierContact.objects.filter(supplier=self.supplier, is_primary=True).exclude(pk=self.pk).update(is_primary=False)
        super().save(*args, **kwargs)

class SupplierNote(models.Model):
    """Notes and comments about suppliers"""
    supplier = models.ForeignKey(Supplier, on_delete=models.CASCADE, related_name='supplier_notes')
    note = models.TextField()
    created_by = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.SET_NULL, null=True, related_name='created_supplier_notes')
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    class Meta:
        ordering = ['-created_at']
    
    def __str__(self):
        return f"Note for {self.supplier.name} by {self.created_by}"

class SupplierDocument(models.Model):
    """Documents related to suppliers (contracts, agreements, etc.)"""
    DOCUMENT_TYPES = [
        ('contract', 'Contract'),
        ('agreement', 'Agreement'),
        ('license', 'License'),
        ('certificate', 'Certificate'),
        ('other', 'Other'),
    ]
    
    supplier = models.ForeignKey(Supplier, on_delete=models.CASCADE, related_name='documents')
    document_type = models.CharField(max_length=20, choices=DOCUMENT_TYPES, default='other')
    title = models.CharField(max_length=200)
    file = models.FileField(upload_to='supplier_documents/%Y/%m/%d/')
    description = models.TextField(blank=True)
    valid_from = models.DateField(null=True, blank=True)
    valid_to = models.DateField(null=True, blank=True)
    uploaded_by = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.SET_NULL, null=True)
    uploaded_at = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        ordering = ['-valid_to', '-uploaded_at']
    
    def __str__(self):
        return f"{self.get_document_type_display()}: {self.title}"
    
    def is_valid(self):
        """Check if the document is currently valid"""
        today = timezone.now().date()
        if self.valid_from and self.valid_from > today:
            return False
        if self.valid_to and self.valid_to < today:
            return False
        return True
