from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from django.http import JsonResponse
from django.db import transaction
from django.db.models import Q
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger
import json
from decimal import Decimal
from pytz import timezone as pytz_timezone

from .models import Sale, SaleItem, Payment
from .forms import SaleForm, QuickCustomerForm, PaymentForm, SaleSearchForm
from products.models import Product
from customers.models import Customer
from core.models import AuditLog, CompanySettings, Notification
from django.utils import timezone

@login_required
def sale_list(request):
    """
    List all sales with filtering and pagination
    """
    # Base queryset filtered by branch if not admin and exclude cancelled sales by default
    if request.user.role == 'admin':
        sales = Sale.objects.exclude(payment_status='cancelled').select_related('customer', 'cashier', 'branch')
    else:
        sales = Sale.objects.filter(branch=request.user.branch).exclude(payment_status='cancelled').select_related('customer', 'cashier', 'branch')
    
    # Initialize search form
    form = SaleSearchForm(request.GET)
    
    # Apply filters if form is valid
    if form.is_valid():
        start_date = form.cleaned_data.get('start_date')
        end_date = form.cleaned_data.get('end_date')
        customer = form.cleaned_data.get('customer')
        invoice = form.cleaned_data.get('invoice')
        payment_status = form.cleaned_data.get('payment_status')
        
        if start_date:
            sales = sales.filter(created_at__date__gte=start_date)
        
        if end_date:
            # Add 1 day to include the end date
            from datetime import timedelta
            end_date = end_date + timedelta(days=1)
            sales = sales.filter(created_at__date__lt=end_date)
        
        if customer:
            sales = sales.filter(
                Q(customer__first_name__icontains=customer) | 
                Q(customer__last_name__icontains=customer)
            )
        
        if invoice:
            sales = sales.filter(invoice_number__icontains=invoice)
        
        if payment_status:
            sales = sales.filter(payment_status=payment_status)
    
    # Order by most recent
    sales = sales.order_by('-created_at')
    
    # Pagination - 20 items per page
    page = request.GET.get('page', 1)
    paginator = Paginator(sales, 20)
    
    try:
        sales_page = paginator.page(page)
    except PageNotAnInteger:
        sales_page = paginator.page(1)
    except EmptyPage:
        sales_page = paginator.page(paginator.num_pages)
    
    context = {
        'sales': sales_page,  # Use paginated sales
        'form': form,
    }
    return render(request, 'sales/sale_list.html', context)

@login_required
def new_sale(request):
    """
    Create a new sale
    """
    # Get or set the branch
    if request.user.role == 'admin' and not request.user.branch:
        messages.warning(request, 'Please select a branch to continue.')
        # Redirect to branch selection or dashboard
        return redirect('dashboard')
    
    branch = request.user.branch
    
    # Initialize forms
    customer_form = QuickCustomerForm(initial={'branch': branch.id})
    
    # Get company settings for receipt
    company_settings = CompanySettings.get_settings()
    
    context = {
        'branch': branch,
        'customer_form': customer_form,
        'company_settings': company_settings,
        'cashier': request.user,
    }
    return render(request, 'sales/new_sale.html', context)

@login_required
def product_search(request):
    """
    API endpoint for searching products during sale
    """
    query = request.GET.get('q', '')
    
    # Filter products by branch and active status
    products = Product.objects.filter(
        branch=request.user.branch,
        is_active=True
    )
    
    # Apply search filter
    if query:
        products = products.filter(
            Q(name__icontains=query) | 
            Q(sku__icontains=query) | 
            Q(barcode__icontains=query)
        )
    
    # Limit to max 10 results
    products = products[:10]
    
    # Format for JSON response
    results = []
    for product in products:
        results.append({
            'id': product.id,
            'name': product.name,
            'price': float(product.price),
            'stock': product.quantity,
            'sku': product.sku,
            'barcode': product.barcode
        })
    
    return JsonResponse({'results': results})

@transaction.atomic
def save_sale(request):
    """
    API endpoint for saving a sale
    Ajax API that accepts requests with proper CSRF token
    """
    # Check if user is authenticated
    if not request.user.is_authenticated:
        return JsonResponse({'status': 'error', 'message': 'Authentication required'}, status=403)
    
    if request.method != 'POST':
        return JsonResponse({'status': 'error', 'message': 'Invalid request method'})
    
    try:
        data = json.loads(request.body)
        
        # Extract sale data
        customer_id = data.get('customer_id')
        items = data.get('items', [])
        discount_amount = Decimal(data.get('discount_amount', 0))
        payment_amount = Decimal(data.get('payment_amount', 0))
        payment_method = data.get('payment_method', 'cash')
        
        if not items:
            return JsonResponse({
                'status': 'error', 
                'message': 'No items in the sale'
            })
        
        # Get customer or set to None
        customer = None
        if customer_id:
            customer = get_object_or_404(Customer, id=customer_id)
        
        # Handle custom date/time
        created_at = data.get('created_at')
        if created_at:
            # Parse and convert to Africa/Nairobi timezone
            from datetime import datetime
            import pytz
            nairobi_tz = pytz_timezone('Africa/Nairobi')
            # Expecting format 'YYYY-MM-DDTHH:MM' or 'YYYY-MM-DDTHH:MM:SS'
            try:
                if len(created_at) == 16:
                    dt = datetime.strptime(created_at, '%Y-%m-%dT%H:%M')
                else:
                    dt = datetime.strptime(created_at, '%Y-%m-%dT%H:%M:%S')
                created_at = nairobi_tz.localize(dt)
            except Exception:
                created_at = timezone.now()
        else:
            created_at = timezone.now()
        
        # Create the sale
        sale = Sale.objects.create(
            customer=customer,
            cashier=request.user,
            branch=request.user.branch,
            discount_amount=discount_amount,
            payment_status='pending',
            created_at=created_at
        )
        
        # Add sale items
        for item_data in items:
            product = get_object_or_404(Product, id=item_data['product_id'])
            quantity = int(item_data['quantity'])
            price = Decimal(item_data['price'])
            
            # Ensure we have enough stock
            if product.quantity < quantity:
                return JsonResponse({
                    'status': 'error',
                    'message': f'Not enough stock for {product.name}. Available: {product.quantity}'
                })
            
            # Create sale item
            SaleItem.objects.create(
                sale=sale,
                product=product,
                quantity=quantity,
                price=price
            )
        
        # Recalculate sale totals
        sale.calculate_totals()
        sale.save()
        
        # Add payment if provided
        if payment_amount > 0:
            Payment.objects.create(
                sale=sale,
                amount=payment_amount,
                payment_method=payment_method
            )
        
        # Log the sale
        AuditLog.log_action(
            user=request.user,
            action='create',
            entity_type='sale',
            entity_id=sale.id,
            details=f"Created sale with invoice #{sale.invoice_number}",
            ip_address=request.META.get('REMOTE_ADDR')
        )
        
        return JsonResponse({
            'status': 'success',
            'sale_id': sale.id,
            'invoice_number': sale.invoice_number,
            'total_amount': float(sale.total_amount),
            'receipt_url': f'/sales/{sale.id}/receipt/'
        })
        
    except Exception as e:
        return JsonResponse({'status': 'error', 'message': str(e)})

@login_required
def sale_detail(request, sale_id):
    """
    View sale details
    """
    # Get sale, ensuring branch access
    if request.user.role == 'admin':
        sale = get_object_or_404(Sale, id=sale_id)
    else:
        sale = get_object_or_404(Sale, id=sale_id, branch=request.user.branch)
    
    # Get sale items and payments
    items = sale.saleitems.all().select_related('product')
    payments = sale.payments.all().order_by('-payment_date')
    
    context = {
        'sale': sale,
        'items': items,
        'payments': payments,
        'company_settings': CompanySettings.get_settings(),
    }
    return render(request, 'sales/sale_detail.html', context)

@login_required
def add_payment(request, sale_id):
    """
    Add a payment to a sale
    """
    # Get sale, ensuring branch access
    if request.user.role == 'admin':
        sale = get_object_or_404(Sale, id=sale_id)
    else:
        sale = get_object_or_404(Sale, id=sale_id, branch=request.user.branch)
    
    if request.method == 'POST':
        form = PaymentForm(request.POST)
        if form.is_valid():
            payment = form.save(commit=False)
            payment.sale = sale
            payment.save()
            
            # Log the payment
            AuditLog.log_action(
                user=request.user,
                action='create',
                entity_type='payment',
                entity_id=payment.id,
                details=f"Added payment of {payment.amount} to sale #{sale.invoice_number}",
                ip_address=request.META.get('REMOTE_ADDR')
            )
            
            messages.success(request, f'Payment of {payment.amount} added successfully.')
            return redirect('sale_detail', sale_id=sale.id)
    else:
        # Set initial amount to balance due
        form = PaymentForm(initial={'amount': sale.balance_due})
    
    context = {
        'form': form,
        'sale': sale,
    }
    return render(request, 'sales/payment_form.html', context)

@login_required
def sale_receipt(request, sale_id):
    """
    View/print receipt for a sale
    """
    # Get sale, ensuring branch access
    if request.user.role == 'admin':
        sale = get_object_or_404(Sale, id=sale_id)
    else:
        sale = get_object_or_404(Sale, id=sale_id, branch=request.user.branch)
    
    # Get sale items and payments
    items = sale.saleitems.all().select_related('product')
    payments = sale.payments.all().order_by('-payment_date')
    
    context = {
        'sale': sale,
        'items': items,
        'payments': payments,
        'company_settings': CompanySettings.get_settings(),
    }
    return render(request, 'sales/receipt.html', context)

@login_required
def cancel_sale(request, sale_id):
    """
    Cancel a sale and return items to inventory
    Only admin users can cancel sales, while cashiers can only view sale details
    """
    # Get sale, ensuring branch access
    if request.user.role == 'admin':
        sale = get_object_or_404(Sale, id=sale_id)
    else:
        sale = get_object_or_404(Sale, id=sale_id, branch=request.user.branch)
        
    # Only admins can cancel sales
    if request.user.role != 'admin':
        messages.error(request, 'Only administrators can cancel sales. Please contact your administrator if this sale needs to be cancelled.')
        return redirect('sale_detail', sale_id=sale.id)
    
    # Only allow cancellation if the sale is pending or partially paid
    if sale.payment_status not in ['pending', 'partial', 'paid']:
        messages.error(request, 'This sale cannot be cancelled.')
        return redirect('sale_detail', sale_id=sale.id)
    
    if request.method == 'POST':
        with transaction.atomic():
            # Return items to inventory
            for item in sale.saleitems.all():
                # Add the quantities back to inventory
                item.product.adjust_stock(
                    'add',
                    item.quantity,
                    user=request.user,
                    reason=f"Sale cancelled - Invoice #{sale.invoice_number}"
                )
            
            # Mark sale as cancelled
            sale.payment_status = 'cancelled'
            sale.save()
            
            # Log the cancellation
            AuditLog.log_action(
                user=request.user,
                action='update',
                entity_type='sale',
                entity_id=sale.id,
                details=f"Cancelled sale with invoice #{sale.invoice_number}",
                ip_address=request.META.get('REMOTE_ADDR')
            )
            
            # Create notification for all admins about the sale cancellation
            Notification.create_sale_cancellation_notification(
                sale=sale,
                user=request.user,
                branch=sale.branch
            )
            
            messages.success(request, f'Sale {sale.invoice_number} has been cancelled and items returned to inventory.')
            return redirect('sale_list')
    
    context = {
        'sale': sale,
    }
    return render(request, 'sales/sale_confirm_cancel.html', context)
