from django.shortcuts import render, get_object_or_404, redirect
from django.views.generic import ListView, CreateView, UpdateView, DetailView, DeleteView, TemplateView
from django.urls import reverse_lazy, reverse
from django.contrib import messages
from django.contrib.auth.mixins import LoginRequiredMixin, UserPassesTestMixin
from django.contrib.auth.decorators import login_required
from django.http import JsonResponse, HttpResponseForbidden, HttpResponse
from django.db import transaction
from django.utils import timezone
from django.template.loader import render_to_string, get_template
from django.core.mail import send_mail
from django.conf import settings
from weasyprint import HTML
from io import BytesIO

from .models import Quotation, QuotationItem, QuotationDocument, QuotationNote
from .forms import QuotationForm, QuotationItemFormSet, QuotationDocumentForm, QuotationNoteForm
from suppliers.models import Supplier


def quotation_print(request, pk):
    """Generate a PDF of the quotation using WeasyPrint"""
    quotation = get_object_or_404(Quotation, pk=pk)
    
    # Get company settings if available
    company_settings = getattr(settings, 'COMPANY_SETTINGS', {})
    
    # Prepare context
    context = {
        'quotation': quotation,
        'items': quotation.items.all(),
        'company_settings': company_settings,
        'now': timezone.now().strftime('%Y-%m-%d %H:%M:%S'),
        'is_print_view': True
    }
    
    # If download parameter is present, force download
    download = request.GET.get('download', '').lower() == '1'
    
    if download or 'download' in request.GET:
        # Render HTML
        html_string = render_to_string('quotations/quotation_print.html', context)
        
        # Create PDF response
        response = HttpResponse(content_type='application/pdf')
        filename = f'quotation_{quotation.quotation_number}.pdf'.replace('/', '_')
        response['Content-Disposition'] = f'attachment; filename="{filename}"'
        
        # Generate PDF using WeasyPrint
        HTML(string=html_string).write_pdf(response)
        return response
    
    # If no download parameter, render HTML for printing
    return render(request, 'quotations/quotation_print.html', context)

class QuotationListView(LoginRequiredMixin, ListView):
    model = Quotation
    template_name = 'quotations/quotation_list.html'
    context_object_name = 'quotations'
    paginate_by = 20
    
    def get_queryset(self):
        queryset = Quotation.objects.all()
        
        # Filter by status
        status = self.request.GET.get('status')
        if status in dict(Quotation.STATUS_CHOICES).keys():
            queryset = queryset.filter(status=status)
            
        # Filter by supplier
        supplier_id = self.request.GET.get('supplier')
        if supplier_id:
            queryset = queryset.filter(supplier_id=supplier_id)
            
        # Search by quotation number or supplier name
        search = self.request.GET.get('search')
        if search:
            queryset = queryset.filter(
                Q(quotation_number__icontains=search) |
                Q(supplier__name__icontains=search)
            )
            
        # Filter by date range
        start_date = self.request.GET.get('start_date')
        end_date = self.request.GET.get('end_date')
        
        if start_date:
            queryset = queryset.filter(issue_date__gte=start_date)
        if end_date:
            queryset = queryset.filter(issue_date__lte=end_date)
            
        # Branch filtering for non-superusers
        if not self.request.user.is_superuser and hasattr(self.request.user, 'branch'):
            queryset = queryset.filter(branch=self.request.user.branch)
            
        return queryset.order_by('-issue_date', '-created_at')
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['status_choices'] = dict(Quotation.STATUS_CHOICES)
        context['current_status'] = self.request.GET.get('status', '')
        context['search_query'] = self.request.GET.get('search', '')
        
        # Add suppliers for filter dropdown
        if hasattr(self.request.user, 'branch'):
            context['suppliers'] = Supplier.objects.filter(branches=self.request.user.branch)
        else:
            context['suppliers'] = Supplier.objects.all()
            
        return context

class QuotationCreateView(LoginRequiredMixin, CreateView):
    model = Quotation
    form_class = QuotationForm
    template_name = 'quotations/quotation_form.html'
    
    def get_form_kwargs(self):
        kwargs = super().get_form_kwargs()
        kwargs['user'] = self.request.user
        return kwargs
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        if self.request.POST:
            context['form'] = self.get_form()
            context['formset'] = QuotationItemFormSet(
                self.request.POST, 
                prefix='items', 
                user=self.request.user,
                instance=self.object if hasattr(self, 'object') else None
            )
        else:
            context['form'] = self.get_form()
            context['formset'] = QuotationItemFormSet(
                prefix='items', 
                user=self.request.user,
                instance=self.object if hasattr(self, 'object') else None
            )
        return context
    
    def form_valid(self, form):
        context = self.get_context_data()
        formset = context['formset']
        
        with transaction.atomic():
            # Set the created_by user
            form.instance.created_by = self.request.user
            
            # Set the supplier based on the user's branch
            if hasattr(self.request.user, 'branch'):
                # Get or create a default supplier for the branch
                supplier, created = Supplier.objects.get_or_create(
                    name=f"Default Supplier - {self.request.user.branch.name}",
                    defaults={
                        'company_name': f"Default Supplier - {self.request.user.branch.name}",
                        'email': 'supplier@example.com',
                        'phone': '123-456-7890',
                        'address_line1': 'Default Address',
                        'is_active': True
                    }
                )
                
                # Add the branch to the supplier's branches if not already added
                if self.request.user.branch not in supplier.branches.all():
                    supplier.branches.add(self.request.user.branch)
                
                form.instance.supplier = supplier
                
                if created:
                    messages.info(self.request, f'Created default supplier for your branch: {supplier.name}')
            else:
                # If user has no branch, create a default supplier without branch association
                supplier, created = Supplier.objects.get_or_create(
                    name="Default Supplier",
                    defaults={
                        'company_name': 'Default Supplier',
                        'email': 'supplier@example.com',
                        'phone': '123-456-7890',
                        'address_line1': 'Default Address',
                        'is_active': True
                    }
                )
                form.instance.supplier = supplier
                
                if created:
                    messages.info(self.request, 'Created default supplier for your quotation')
            
            self.object = form.save()
            
            if formset.is_valid():
                formset.instance = self.object
                formset.save()
                
                # If print parameter is present, redirect to print view
                if 'print' in self.request.POST:
                    return redirect('quotations:quotation_print', pk=self.object.pk)
                
                # Add success message with actions
                print_url = reverse('quotations:quotation_print', args=[self.object.pk])
                success_message = (
                    f'<div class="d-flex align-items-center">'
                    f'  <i class="fas fa-check-circle text-success me-2"></i>'
                    f'  <div>'
                    f'    <div class="fw-bold">Quotation {self.object.quotation_number} created successfully!</div>'
                    f'    <div class="text-muted small mt-1">What would you like to do next?</div>'
                    f'  </div>'
                    f'</div>'
                    f'<div class="mt-2">'
                    f'  <a href="{print_url}" class="btn btn-sm btn-outline-primary me-2" target="_blank">'
                    f'    <i class="fas fa-print me-1"></i> Print Quotation'
                    f'  </a>'
                    f'  <a href="{print_url}?download=1" class="btn btn-sm btn-outline-success me-2">'
                    f'    <i class="fas fa-download me-1"></i> Download PDF'
                    f'  </a>'
                    f'  <a href="{reverse('quotations:quotation_detail', args=[self.object.pk])}" class="btn btn-sm btn-outline-secondary">'
                    f'    <i class="fas fa-eye me-1"></i> View Details'
                    f'  </a>'
                    f'</div>'
                )
                messages.success(self.request, success_message, extra_tags='safe')
                
                if 'save_and_continue' in self.request.POST:
                    return redirect('quotations:quotation_create')
                return redirect('quotations:quotation_detail', pk=self.object.pk)
            
        return self.render_to_response(self.get_context_data(form=form))

class QuotationUpdateView(LoginRequiredMixin, UserPassesTestMixin, UpdateView):
    model = Quotation
    form_class = QuotationForm
    template_name = 'quotations/quotation_form.html'
    
    def test_func(self):
        quotation = self.get_object()
        return self.request.user == quotation.created_by or self.request.user.is_superuser
    
    def get_form_kwargs(self):
        kwargs = super().get_form_kwargs()
        kwargs['user'] = self.request.user
        return kwargs
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        if self.request.POST:
            context['form'] = self.get_form()
            context['formset'] = QuotationItemFormSet(
                self.request.POST, 
                instance=self.object, 
                prefix='items',
                user=self.request.user
            )
        else:
            context['form'] = self.get_form()
            context['formset'] = QuotationItemFormSet(
                instance=self.object, 
                prefix='items',
                user=self.request.user
            )
            
        # Add document form
        context['document_form'] = QuotationDocumentForm()
        context['note_form'] = QuotationNoteForm()
            
        return context
    
    def form_valid(self, form):
        context = self.get_context_data()
        formset = context['formset']
        
        with transaction.atomic():
            self.object = form.save()
            
            if formset.is_valid():
                formset.instance = self.object
                formset.save()
                
                messages.success(self.request, 'Quotation updated successfully.')
                return redirect('quotations:quotation_detail', pk=self.object.pk)
            
        return self.render_to_response(self.get_context_data(form=form))

class QuotationDetailView(LoginRequiredMixin, DetailView):
    model = Quotation
    template_name = 'quotations/quotation_detail.html'
    context_object_name = 'quotation'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['document_form'] = QuotationDocumentForm()
        context['note_form'] = QuotationNoteForm()
        return context

class QuotationDeleteView(LoginRequiredMixin, UserPassesTestMixin, DeleteView):
    model = Quotation
    template_name = 'quotations/quotation_confirm_delete.html'
    success_url = reverse_lazy('quotations:quotation_list')
    
    def test_func(self):
        quotation = self.get_object()
        return (self.request.user == quotation.created_by or 
                self.request.user.is_superuser)
    
    def delete(self, request, *args, **kwargs):
        messages.success(request, 'Quotation deleted successfully.')
        return super().delete(request, *args, **kwargs)

@login_required
def change_quotation_status(request, pk):
    quotation = get_object_or_404(Quotation, pk=pk)
    
    # Check permissions
    if not (request.user == quotation.created_by or request.user.is_superuser):
        return HttpResponseForbidden()
    
    if request.method == 'POST':
        new_status = request.POST.get('status')
        
        if new_status in dict(Quotation.STATUS_CHOICES).keys():
            quotation.status = new_status
            quotation.save()
            
            # Add a note about the status change
            QuotationNote.objects.create(
                quotation=quotation,
                note=f"Status changed to {quotation.get_status_display()}",
                created_by=request.user
            )
            
            messages.success(request, f'Quotation status updated to {quotation.get_status_display()}.')
            
            # Send email notification if status is sent or accepted
            if new_status in [Quotation.SENT, Quotation.ACCEPTED]:
                self._send_status_notification(quotation)
        
        return redirect('quotations:quotation_detail', pk=quotation.pk)
    
    return redirect('quotations:quotation_list')

def _send_status_notification(self, quotation):
    """Send email notification about status change"""
    subject = f"Quotation {quotation.quotation_number} - {quotation.get_status_display()}"
    
    context = {
        'quotation': quotation,
        'site_name': 'Your Company Name',
        'status': quotation.get_status_display(),
    }
    
    message = render_to_string('quotations/emails/status_notification.txt', context)
    html_message = render_to_string('quotations/emails/status_notification.html', context)
    
    recipient_list = [quotation.supplier.email]
    if quotation.created_by.email:
        recipient_list.append(quotation.created_by.email)
    
    send_mail(
        subject=subject,
        message=message,
        from_email=settings.DEFAULT_FROM_EMAIL,
        recipient_list=recipient_list,
        html_message=html_message,
        fail_silently=True,
    )

@login_required
def add_quotation_document(request, pk):
    quotation = get_object_or_404(Quotation, pk=pk)
    
    # Check permissions
    if not (request.user == quotation.created_by or request.user.is_superuser):
        return HttpResponseForbidden()
    
    if request.method == 'POST':
        form = QuotationDocumentForm(request.POST, request.FILES)
        if form.is_valid():
            document = form.save(commit=False)
            document.quotation = quotation
            document.uploaded_by = request.user
            document.save()
            
            messages.success(request, 'Document added successfully.')
        else:
            messages.error(request, 'Error adding document. Please check the form.')
    
    return redirect('quotations:quotation_detail', pk=quotation.pk)

@login_required
def add_quotation_note(request, pk):
    quotation = get_object_or_404(Quotation, pk=pk)
    
    if request.method == 'POST':
        form = QuotationNoteForm(request.POST)
        if form.is_valid():
            note = form.save(commit=False)
            note.quotation = quotation
            note.created_by = request.user
            note.save()
            
            messages.success(request, 'Note added successfully.')
        else:
            messages.error(request, 'Error adding note. Please check the form.')
    
    return redirect('quotations:quotation_detail', pk=quotation.pk)
