from django.test import TestCase
from django.urls import reverse
from django.contrib.auth import get_user_model
from branches.models import Branch
from .models import Category, Product, StockMovement

User = get_user_model()

class ProductModelTest(TestCase):
    def setUp(self):
        self.branch = Branch.objects.create(
            name='Test Branch',
            address='123 Test St'
        )
        
        self.category = Category.objects.create(
            name='Test Category',
            branch=self.branch
        )
        
        self.product = Product.objects.create(
            name='Test Product',
            description='Test Description',
            category=self.category,
            branch=self.branch,
            price=10.00,
            cost_price=5.00,
            quantity=20,
            low_stock_threshold=5,
            sku='TEST001',
            barcode='1234567890'
        )
    
    def test_product_creation(self):
        self.assertEqual(Product.objects.count(), 1)
        self.assertEqual(self.product.name, 'Test Product')
        self.assertEqual(self.product.price, 10.00)
        self.assertEqual(self.product.quantity, 20)
    
    def test_product_properties(self):
        self.assertFalse(self.product.is_low_stock)
        self.assertEqual(self.product.profit_margin, 100.0)  # (10-5)/5 * 100 = 100%
    
    def test_stock_adjustment(self):
        # Add stock
        self.product.adjust_stock('add', 10, reason='Testing stock addition')
        self.assertEqual(self.product.quantity, 30)
        
        # Remove stock
        self.product.adjust_stock('remove', 15, reason='Testing stock removal')
        self.assertEqual(self.product.quantity, 15)
        
        # Set stock
        self.product.adjust_stock('set', 25, reason='Testing stock setting')
        self.assertEqual(self.product.quantity, 25)
        
        # Check stock movement records
        self.assertEqual(StockMovement.objects.count(), 3)

class ProductViewTest(TestCase):
    def setUp(self):
        self.user = User.objects.create_user(
            email='testuser@example.com',
            password='testpassword',
            first_name='Test',
            last_name='User',
            role='admin'
        )
        self.client.login(username='testuser@example.com', password='testpassword')
        
        self.branch = Branch.objects.create(
            name='Test Branch',
            address='123 Test St'
        )
        
        self.category = Category.objects.create(
            name='Test Category',
            branch=self.branch
        )
        
        self.product = Product.objects.create(
            name='Test Product',
            category=self.category,
            branch=self.branch,
            price=10.00,
            quantity=20,
            sku='TEST001'
        )
    
    def test_product_list_view(self):
        response = self.client.get(reverse('product_list'))
        self.assertEqual(response.status_code, 200)
        self.assertContains(response, 'Test Product')
    
    def test_product_create_view(self):
        response = self.client.post(reverse('product_add'), {
            'name': 'New Product',
            'category': self.category.id,
            'branch': self.branch.id,
            'price': 15.00,
            'cost_price': 7.50,
            'quantity': 30,
            'low_stock_threshold': 5,
            'sku': 'NEW001',
            'is_active': True
        })
        self.assertRedirects(response, reverse('product_list'))
        self.assertEqual(Product.objects.count(), 2)
