from django.test import TestCase
from django.urls import reverse
from django.contrib.auth import get_user_model
from .models import Branch
from licenses.models import License
from datetime import timedelta
from django.utils import timezone

User = get_user_model()

class BranchModelTest(TestCase):
    def setUp(self):
        self.branch = Branch.objects.create(
            name='Test Branch',
            address='123 Test St',
            phone='123-456-7890',
            email='branch@example.com'
        )
    
    def test_branch_creation(self):
        self.assertEqual(Branch.objects.count(), 1)
        self.assertEqual(self.branch.name, 'Test Branch')
        self.assertTrue(self.branch.is_active)
    
    def test_license_validation(self):
        # Initially no license
        self.assertFalse(self.branch.has_valid_license())
        
        # Create a valid license
        tomorrow = timezone.now().date() + timedelta(days=1)
        License.objects.create(
            branch=self.branch,
            license_key='TEST-LICENSE-KEY',
            expiry_date=tomorrow,
            is_active=True
        )
        
        self.assertTrue(self.branch.has_valid_license())
        
        # Create an expired license
        yesterday = timezone.now().date() - timedelta(days=1)
        License.objects.create(
            branch=self.branch,
            license_key='EXPIRED-LICENSE-KEY',
            expiry_date=yesterday,
            is_active=True
        )
        
        # Should still return True because we have a valid license
        self.assertTrue(self.branch.has_valid_license())

class BranchViewTest(TestCase):
    def setUp(self):
        self.admin_user = User.objects.create_user(
            email='admin@example.com',
            password='testpassword',
            first_name='Admin',
            last_name='User',
            role='admin'
        )
        self.client.login(username='admin@example.com', password='testpassword')
        
        self.branch = Branch.objects.create(
            name='Test Branch',
            address='123 Test St',
            phone='123-456-7890',
            email='branch@example.com'
        )
    
    def test_branch_list_view(self):
        response = self.client.get(reverse('branch_list'))
        self.assertEqual(response.status_code, 200)
        self.assertContains(response, 'Test Branch')
    
    def test_branch_create_view(self):
        response = self.client.post(reverse('branch_add'), {
            'name': 'New Branch',
            'address': '456 New St',
            'phone': '987-654-3210',
            'email': 'new@example.com',
            'is_active': True
        })
        self.assertRedirects(response, reverse('branch_list'))
        self.assertEqual(Branch.objects.count(), 2)
