from django.shortcuts import render, HttpResponse, get_object_or_404, redirect
from django.contrib.auth.decorators import login_required, user_passes_test
from django.contrib import messages
from django.db.models import Sum, Count, Avg, F, ExpressionWrapper, DecimalField, Q
from django.db.models.functions import TruncDate, TruncMonth, TruncYear
from django.utils import timezone
from django.template.loader import render_to_string
import csv
import io
import json
from datetime import datetime, timedelta
from weasyprint import HTML
from io import BytesIO
from django.template.loader import render_to_string

from sales.models import Sale, SaleItem, Payment
from products.models import Product, StockMovement, Category
from customers.models import Customer
from accounts.models import User
from core.models import CompanySettings
from core.views import CustomJSONEncoder

def is_admin_or_manager(user):
    """
    Check if user is an admin or manager
    """
    return user.is_authenticated and (user.role in ['admin', 'manager'])

def can_view_reports(user):
    """
    Check if user can view any reports (admin, manager, cashier)
    """
    return user.is_authenticated

@login_required
@user_passes_test(can_view_reports)
def reports_dashboard(request):
    """
    Reports dashboard with links to different reports
    """
    # If user is a cashier, redirect them directly to their own report
    if request.user.role == 'cashier':
        return redirect('cashier_detail_report', cashier_id=request.user.id)
    
    context = {
        'title': 'Reports Dashboard',
    }
    return render(request, 'reports/reports.html', context)

@login_required
@user_passes_test(is_admin_or_manager)
def sales_report(request):
    """
    Sales report view
    """
    # Get date range parameters
    start_date_str = request.GET.get('start_date')
    end_date_str = request.GET.get('end_date')
    group_by = request.GET.get('group_by', 'day')
    
    # Set default date range to last 30 days if not specified
    today = timezone.now().date()
    if start_date_str:
        start_date = datetime.strptime(start_date_str, '%Y-%m-%d').date()
    else:
        start_date = today - timedelta(days=30)
    
    if end_date_str:
        end_date = datetime.strptime(end_date_str, '%Y-%m-%d').date()
    else:
        end_date = today
    
    # Base queryset filtered by branch if not admin and by date range
    if request.user.role == 'admin':
        sales_queryset = Sale.objects.filter(
            created_at__date__gte=start_date,
            created_at__date__lte=end_date
        )
    else:
        sales_queryset = Sale.objects.filter(
            branch=request.user.branch,
            created_at__date__gte=start_date,
            created_at__date__lte=end_date
        )
    
    # Group data by selected time period
    if group_by == 'month':
        sales_over_time = sales_queryset.annotate(
            period=TruncMonth('created_at')
        ).values('period').annotate(
            total_sales=Count('id'),
            total_amount=Sum('total_amount')
        ).order_by('period')
    elif group_by == 'year':
        sales_over_time = sales_queryset.annotate(
            period=TruncYear('created_at')
        ).values('period').annotate(
            total_sales=Count('id'),
            total_amount=Sum('total_amount')
        ).order_by('period')
    else:  # Default to day
        sales_over_time = sales_queryset.annotate(
            period=TruncDate('created_at')
        ).values('period').annotate(
            total_sales=Count('id'),
            total_amount=Sum('total_amount')
        ).order_by('period')
    
    # Calculate sales by payment method
    sales_by_payment = Payment.objects.filter(
        sale__in=sales_queryset
    ).values('payment_method').annotate(
        total=Sum('amount')
    ).order_by('-total')
    
    # Calculate sales by cashier
    sales_by_cashier = sales_queryset.values(
        'cashier__first_name', 'cashier__last_name'
    ).annotate(
        total_sales=Count('id'),
        total_amount=Sum('total_amount')
    ).order_by('-total_amount')
    
    # Calculate sales by product category
    sales_by_category = SaleItem.objects.filter(
        sale__in=sales_queryset
    ).values(
        'product__category__name'
    ).annotate(
        total_quantity=Sum('quantity'),
        total_amount=Sum('total')
    ).order_by('-total_amount')
    
    # Get top selling products
    top_products = SaleItem.objects.filter(
        sale__in=sales_queryset
    ).values(
        'product__name'
    ).annotate(
        total_quantity=Sum('quantity'),
        total_amount=Sum('total')
    ).order_by('-total_quantity')[:10]
    
    # Summary statistics
    summary = {
        'total_sales': sales_queryset.count(),
        'total_amount': sales_queryset.aggregate(total=Sum('total_amount'))['total'] or 0,
        'average_sale': sales_queryset.aggregate(avg=Avg('total_amount'))['avg'] or 0,
        'total_items_sold': SaleItem.objects.filter(sale__in=sales_queryset).aggregate(total=Sum('quantity'))['total'] or 0,
    }
    
    context = {
        'title': 'Sales Report',
        'start_date': start_date,
        'end_date': end_date,
        'group_by': group_by,
        'sales_over_time': list(sales_over_time),
        'sales_by_payment': list(sales_by_payment),
        'sales_by_cashier': list(sales_by_cashier),
        'sales_by_category': list(sales_by_category),
        'top_products': list(top_products),
        'summary': summary,
    }
    return render(request, 'reports/sales_report.html', context)

@login_required
@user_passes_test(is_admin_or_manager)
def inventory_report(request):
    """
    Inventory report view
    """
    # Base queryset filtered by branch if not admin
    if request.user.role == 'admin':
        products = Product.objects.all().select_related('category', 'branch')
    else:
        products = Product.objects.filter(branch=request.user.branch).select_related('category', 'branch')
    
    # Get category filter
    category_id = request.GET.get('category')
    if category_id:
        products = products.filter(category_id=category_id)
    
    # Calculate inventory values
    products = products.annotate(
        stock_value=ExpressionWrapper(
            F('quantity') * F('price'),
            output_field=DecimalField()
        ),
        cost_value=ExpressionWrapper(
            F('quantity') * F('cost_price'),
            output_field=DecimalField()
        ),
    )
    
    # Get low stock products
    low_stock_products = products.filter(quantity__lt=F('low_stock_threshold'))
    
    # Calculate summary statistics
    total_products = products.count()
    total_stock_value = sum(product.stock_value for product in products)
    total_cost_value = sum(product.cost_value for product in products)
    
    # Calculate potential profit
    potential_profit = total_stock_value - total_cost_value
    
    # Stock movements in the last 30 days
    today = timezone.now().date()
    thirty_days_ago = today - timedelta(days=30)
    
    recent_stock_movements = StockMovement.objects.filter(
        timestamp__date__gte=thirty_days_ago,
        product__in=products
    ).select_related('product').order_by('-timestamp')[:20]
    
    context = {
        'title': 'Inventory Report',
        'products': products,
        'low_stock_products': low_stock_products,
        'total_products': total_products,
        'total_stock_value': total_stock_value,
        'total_cost_value': total_cost_value,
        'potential_profit': potential_profit,
        'recent_stock_movements': recent_stock_movements,
    }
    return render(request, 'reports/inventory_report.html', context)

@login_required
@user_passes_test(is_admin_or_manager)
def customers_report(request):
    """
    Customer analytics report
    """
    # Base queryset filtered by branch if not admin
    if request.user.role == 'admin':
        customers = Customer.objects.all()
    else:
        customers = Customer.objects.filter(branch=request.user.branch)
    
    # Calculate customer metrics
    customer_metrics = []
    
    for customer in customers:
        sales = Sale.objects.filter(customer=customer)
        
        total_purchases = sales.count()
        if total_purchases == 0:
            continue  # Skip customers with no purchases
        
        total_spent = sales.aggregate(total=Sum('total_amount'))['total'] or 0
        average_purchase = total_spent / total_purchases if total_purchases > 0 else 0
        last_purchase = sales.order_by('-created_at').first()
        
        customer_metrics.append({
            'id': customer.id,
            'name': customer.full_name,
            'email': customer.email,
            'phone': customer.phone,
            'total_purchases': total_purchases,
            'total_spent': total_spent,
            'average_purchase': average_purchase,
            'last_purchase_date': last_purchase.created_at if last_purchase else None,
        })
    
    # Sort by total spent (descending)
    customer_metrics.sort(key=lambda x: x['total_spent'], reverse=True)
    
    # Calculate top-level statistics
    total_customers = len(customer_metrics)
    total_revenue = sum(metric['total_spent'] for metric in customer_metrics)
    average_customer_value = total_revenue / total_customers if total_customers > 0 else 0
    
    context = {
        'title': 'Customer Report',
        'customer_metrics': customer_metrics[:100],  # Limit to top 100 for performance
        'total_customers': total_customers,
        'total_revenue': total_revenue,
        'average_customer_value': average_customer_value,
    }
    return render(request, 'reports/customers_report.html', context)

@login_required
@user_passes_test(is_admin_or_manager)
def cashiers_report(request):
    """
    Cashier performance report
    """
    # Base queryset filtered by branch if not admin
    if request.user.role == 'admin':
        cashiers = User.objects.filter(role__in=['cashier', 'manager'])
    else:
        cashiers = User.objects.filter(branch=request.user.branch, role__in=['cashier', 'manager'])
    
    # Get date range parameters
    start_date_str = request.GET.get('start_date')
    end_date_str = request.GET.get('end_date')
    
    # Set default date range to last 30 days if not specified
    today = timezone.now().date()
    if start_date_str:
        start_date = datetime.strptime(start_date_str, '%Y-%m-%d').date()
    else:
        start_date = today - timedelta(days=30)
    
    if end_date_str:
        end_date = datetime.strptime(end_date_str, '%Y-%m-%d').date()
    else:
        end_date = today
    
    # Calculate cashier metrics
    cashier_metrics = []
    
    for cashier in cashiers:
        # Get sales by this cashier in the date range
        sales = Sale.objects.filter(
            cashier=cashier,
            created_at__date__gte=start_date,
            created_at__date__lte=end_date
        )
        
        # Skip if no sales
        if sales.count() == 0:
            continue
        
        # Calculate metrics
        total_sales = sales.count()
        total_amount = sales.aggregate(total=Sum('total_amount'))['total'] or 0
        average_sale = total_amount / total_sales if total_sales > 0 else 0
        
        # Items per sale
        items_sold = SaleItem.objects.filter(sale__in=sales).aggregate(total=Sum('quantity'))['total'] or 0
        items_per_sale = items_sold / total_sales if total_sales > 0 else 0
        
        cashier_metrics.append({
            'id': cashier.id,
            'name': f"{cashier.first_name} {cashier.last_name}",
            'email': cashier.email,
            'role': cashier.role,
            'total_sales': total_sales,
            'total_amount': total_amount,
            'average_sale': average_sale,
            'items_sold': items_sold,
            'items_per_sale': items_per_sale,
        })
    
    # Sort by total amount (descending)
    cashier_metrics.sort(key=lambda x: x['total_amount'], reverse=True)
    
    context = {
        'title': 'Cashier Performance Report',
        'cashier_metrics': cashier_metrics,
        'start_date': start_date,
        'end_date': end_date,
    }
    return render(request, 'reports/cashiers_report.html', context)

@login_required
@user_passes_test(can_view_reports)  # All authenticated users can view
def cashier_detail_report(request, cashier_id):
    """
    Detailed report for a specific cashier (seller)
    """
    # Get the cashier
    cashier = get_object_or_404(User, id=cashier_id, role__in=['cashier', 'manager'])
    
    # Get date range parameters
    start_date_str = request.GET.get('start_date')
    end_date_str = request.GET.get('end_date')
    
    # Set default date range to last 30 days if not specified
    today = timezone.now().date()
    if start_date_str:
        start_date = datetime.strptime(start_date_str, '%Y-%m-%d').date()
    else:
        start_date = today - timedelta(days=30)
    
    if end_date_str:
        end_date = datetime.strptime(end_date_str, '%Y-%m-%d').date()
    else:
        end_date = today
    
    # Get sales by this cashier in the date range
    sales = Sale.objects.filter(
        cashier=cashier,
        created_at__date__gte=start_date,
        created_at__date__lte=end_date
    ).order_by('-created_at')
    
    # Calculate basic metrics
    total_sales = sales.count()
    total_amount = sales.aggregate(total=Sum('total_amount'))['total'] or 0
    average_sale = total_amount / total_sales if total_sales > 0 else 0
    items_sold = SaleItem.objects.filter(sale__in=sales).aggregate(total=Sum('quantity'))['total'] or 0
    items_per_sale = items_sold / total_sales if total_sales > 0 else 0
    
    metrics = {
        'total_sales': total_sales,
        'total_amount': total_amount,
        'average_sale': average_sale,
        'items_sold': items_sold,
        'items_per_sale': items_per_sale,
    }
    
    # Daily sales data for chart
    daily_sales = sales.annotate(
        date=TruncDate('created_at')
    ).values('date').annotate(
        total=Sum('total_amount')
    ).order_by('date')
    
    # Prepare chart data
    daily_sales_dates = [item['date'].strftime('%Y-%m-%d') for item in daily_sales]
    daily_sales_amounts = [float(item['total']) for item in daily_sales]
    
    # Sales by product category
    category_sales = SaleItem.objects.filter(
        sale__in=sales
    ).values(
        'product__category__name'
    ).annotate(
        total=Sum('total')
    ).order_by('-total')
    
    category_labels = [item['product__category__name'] or 'Uncategorized' for item in category_sales]
    category_data = [float(item['total']) for item in category_sales]
    
    # Sales by payment method
    payment_methods = Payment.objects.filter(
        sale__in=sales
    ).values(
        'payment_method'
    ).annotate(
        total=Sum('amount')
    ).order_by('-total')
    
    payment_labels = [item['payment_method'].capitalize() for item in payment_methods]
    payment_data = [float(item['total']) for item in payment_methods]
    
    # Add sale item count to each sale for display
    for sale in sales:
        sale.items_count = SaleItem.objects.filter(sale=sale).aggregate(total=Sum('quantity'))['total'] or 0
    
    context = {
        'title': f'Cashier Report: {cashier.first_name} {cashier.last_name}',
        'cashier': cashier,
        'metrics': metrics,
        'sales': sales,
        'start_date': start_date,
        'end_date': end_date,
        'daily_sales_labels': json.dumps(daily_sales_dates, cls=CustomJSONEncoder),
        'daily_sales_data': json.dumps(daily_sales_amounts, cls=CustomJSONEncoder),
        'category_labels': json.dumps(category_labels, cls=CustomJSONEncoder),
        'category_data': json.dumps(category_data, cls=CustomJSONEncoder),
        'payment_labels': json.dumps(payment_labels, cls=CustomJSONEncoder),
        'payment_data': json.dumps(payment_data, cls=CustomJSONEncoder),
    }
    return render(request, 'reports/cashier_detail_report.html', context)

@login_required
@user_passes_test(is_admin_or_manager)
def export_sales_report(request):
    """
    Export sales report to CSV
    """
    # Get date range parameters
    start_date_str = request.GET.get('start_date')
    end_date_str = request.GET.get('end_date')
    
    # Set default date range to last 30 days if not specified
    today = timezone.now().date()
    if start_date_str:
        start_date = datetime.strptime(start_date_str, '%Y-%m-%d').date()
    else:
        start_date = today - timedelta(days=30)
    
    if end_date_str:
        end_date = datetime.strptime(end_date_str, '%Y-%m-%d').date()
    else:
        end_date = today
    
    # Base queryset filtered by branch if not admin and by date range
    if request.user.role == 'admin':
        sales = Sale.objects.filter(
            created_at__date__gte=start_date,
            created_at__date__lte=end_date
        ).select_related('customer', 'cashier', 'branch')
    else:
        sales = Sale.objects.filter(
            branch=request.user.branch,
            created_at__date__gte=start_date,
            created_at__date__lte=end_date
        ).select_related('customer', 'cashier', 'branch')
    
    # Create CSV file
    output = io.StringIO()
    writer = csv.writer(output)
    
    # Write header row
    writer.writerow([
        'Invoice Number', 'Date', 'Customer', 'Cashier', 'Branch',
        'Subtotal', 'Tax', 'Discount', 'Total', 'Status'
    ])
    
    # Write data rows
    for sale in sales:
        writer.writerow([
            sale.invoice_number,
            sale.created_at.strftime('%Y-%m-%d %H:%M'),
            sale.customer.full_name if sale.customer else 'Walk-in Customer',
            f"{sale.cashier.first_name} {sale.cashier.last_name}" if sale.cashier else 'Unknown',
            sale.branch.name,
            sale.subtotal,
            sale.tax_amount,
            sale.discount_amount,
            sale.total_amount,
            sale.payment_status.capitalize()
        ])
    
    # Prepare response
    output.seek(0)
    response = HttpResponse(output.read(), content_type='text/csv')
    response['Content-Disposition'] = f'attachment; filename=sales_report_{start_date}_{end_date}.csv'
    
    return response
    
def export_cashier_sales_report(request, cashier_id):
    """
    Export sales report for a specific cashier to CSV
    """
    # Get the cashier
    cashier = get_object_or_404(User, id=cashier_id, role__in=['cashier', 'manager'])
    
    # Check permissions: only the cashier themselves, a manager from their branch, or an admin can export their report
    if request.user.id != cashier_id and request.user.role != 'admin' and not (request.user.role == 'manager' and request.user.branch == cashier.branch):
        messages.error(request, "You don't have permission to access this report.")
        return redirect('dashboard')
    
    # Get date range parameters
    start_date_str = request.GET.get('start_date')
    end_date_str = request.GET.get('end_date')
    
    # Set default date range to last 30 days if not specified
    today = timezone.now().date()
    if start_date_str:
        start_date = datetime.strptime(start_date_str, '%Y-%m-%d').date()
    else:
        start_date = today - timedelta(days=30)
    
    if end_date_str:
        end_date = datetime.strptime(end_date_str, '%Y-%m-%d').date()
    else:
        end_date = today
    
    # Get sales by this cashier in the date range
    sales = Sale.objects.filter(
        cashier=cashier,
        created_at__date__gte=start_date,
        created_at__date__lte=end_date
    ).select_related('customer', 'branch')
    
    # Create CSV file
    output = io.StringIO()
    writer = csv.writer(output)
    
    # Add report metadata
    company_settings = CompanySettings.objects.first()
    writer.writerow(['Sales Report for ' + cashier.get_full_name()])
    writer.writerow(['Date Range: ' + start_date.strftime('%Y-%m-%d') + ' to ' + end_date.strftime('%Y-%m-%d')])
    writer.writerow(['Generated on: ' + timezone.now().strftime('%Y-%m-%d %H:%M:%S')])
    if company_settings:
        writer.writerow(['Company: ' + company_settings.company_name])
    if cashier.branch:
        writer.writerow(['Branch: ' + cashier.branch.name])
    writer.writerow([])  # Empty row as separator
    
    # Write header row
    writer.writerow([
        'Invoice Number', 'Date', 'Customer', 'Items Count', 
        'Subtotal', 'Tax', 'Discount', 'Total', 'Payment Status'
    ])
    
    # Write data rows
    total_amount = 0
    total_items = 0
    
    for sale in sales:
        # Get items count for this sale
        items_count = SaleItem.objects.filter(sale=sale).aggregate(total=Sum('quantity'))['total'] or 0
        total_items += items_count
        total_amount += sale.total_amount
        
        writer.writerow([
            sale.invoice_number,
            sale.created_at.strftime('%Y-%m-%d %H:%M'),
            sale.customer.full_name if sale.customer else 'Walk-in Customer',
            items_count,
            sale.subtotal,
            sale.tax_amount,
            sale.discount_amount,
            sale.total_amount,
            sale.payment_status.capitalize()
        ])
    
    # Write summary information
    writer.writerow([])  # Empty row as separator
    writer.writerow(['Summary'])
    writer.writerow(['Total Sales:', sales.count()])
    writer.writerow(['Total Items Sold:', total_items])
    writer.writerow(['Total Revenue:', total_amount])
    writer.writerow(['Average Sale Value:', (total_amount / sales.count() if sales.count() > 0 else 0)])
    
    # Prepare response
    output.seek(0)
    response = HttpResponse(output.read(), content_type='text/csv')
    filename = f"cashier_sales_{cashier.first_name}_{cashier.last_name}_{start_date}_{end_date}.csv"
    response['Content-Disposition'] = f'attachment; filename={filename}'
    
    return response


def export_cashier_sales_pdf_report(request, cashier_id):
    """
    Export sales report for a specific cashier to PDF
    """
    # Get the cashier
    cashier = get_object_or_404(User, id=cashier_id, role__in=['cashier', 'manager'])
    
    # Check permissions: only the cashier themselves, a manager from their branch, or an admin can export their report
    if request.user.id != cashier_id and request.user.role != 'admin' and not (request.user.role == 'manager' and request.user.branch == cashier.branch):
        messages.error(request, "You don't have permission to access this report.")
        return redirect('dashboard')
    
    # Get date range parameters
    start_date_str = request.GET.get('start_date')
    end_date_str = request.GET.get('end_date')
    
    # Set default date range to last 30 days if not specified
    today = timezone.now().date()
    if start_date_str:
        start_date = datetime.strptime(start_date_str, '%Y-%m-%d').date()
    else:
        start_date = today - timedelta(days=30)
    
    if end_date_str:
        end_date = datetime.strptime(end_date_str, '%Y-%m-%d').date()
    else:
        end_date = today
    
    # Get sales by this cashier in the date range
    sales = Sale.objects.filter(
        cashier=cashier,
        created_at__date__gte=start_date,
        created_at__date__lte=end_date
    ).select_related('customer', 'branch').order_by('-created_at')
    
    # Calculate metrics
    total_sales = sales.count()
    total_amount = sales.aggregate(total=Sum('total_amount'))['total'] or 0
    average_sale = total_amount / total_sales if total_sales > 0 else 0
    items_sold = SaleItem.objects.filter(sale__in=sales).aggregate(total=Sum('quantity'))['total'] or 0
    items_per_sale = items_sold / total_sales if total_sales > 0 else 0
    
    # For each sale, calculate items count
    for sale in sales:
        sale.items_count = SaleItem.objects.filter(sale=sale).aggregate(total=Sum('quantity'))['total'] or 0
    
    # Company settings for header
    company_settings = CompanySettings.objects.first()
    
    # Prepare context for the template
    context = {
        'cashier': cashier,
        'sales': sales,
        'start_date': start_date,
        'end_date': end_date,
        'metrics': {
            'total_sales': total_sales,
            'total_amount': total_amount,
            'average_sale': average_sale,
            'items_sold': items_sold,
            'items_per_sale': items_per_sale
        },
        'company_settings': company_settings,
        'generation_time': timezone.now()
    }
    
    # Render HTML template
    html_string = render_to_string('reports/pdf_template.html', context)
    
    # Configure pdfkit options
    options = {
        'page-size': 'A4',
        'margin-top': '0.5in',
        'margin-right': '0.5in',
        'margin-bottom': '0.5in',
        'margin-left': '0.5in',
        'encoding': 'UTF-8',
        'quiet': ''
    }
    
    # Generate PDF using WeasyPrint
    html = HTML(string=html_string)
    pdf = html.write_pdf()
    
    # Create response
    response = HttpResponse(pdf, content_type='application/pdf')
    filename = f"cashier_sales_{cashier.first_name}_{cashier.last_name}_{start_date}_{end_date}.pdf"
    response['Content-Disposition'] = f'attachment; filename="{filename}"'
    
    return response

@login_required
@user_passes_test(is_admin_or_manager)
def export_inventory_report(request):
    """
    Export inventory report to CSV
    """
    # Base queryset filtered by branch if not admin
    if request.user.role == 'admin':
        products = Product.objects.all().select_related('category', 'branch')
    else:
        products = Product.objects.filter(branch=request.user.branch).select_related('category', 'branch')
    
    # Get category filter
    category_id = request.GET.get('category')
    if category_id:
        products = products.filter(category_id=category_id)
    
    # Create CSV file
    output = io.StringIO()
    writer = csv.writer(output)
    
    # Write header row
    writer.writerow([
        'SKU', 'Barcode', 'Name', 'Category', 'Branch',
        'Quantity', 'Price', 'Cost Price', 'Stock Value', 'Cost Value',
        'Profit Margin (%)', 'Low Stock Threshold', 'Status'
    ])
    
    # Write data rows
    for product in products:
        stock_value = product.quantity * product.price
        cost_value = product.quantity * product.cost_price
        profit_margin = ((product.price - product.cost_price) / product.cost_price * 100) if product.cost_price > 0 else 0
        
        writer.writerow([
            product.sku,
            product.barcode,
            product.name,
            product.category.name if product.category else 'Uncategorized',
            product.branch.name,
            product.quantity,
            product.price,
            product.cost_price,
            stock_value,
            cost_value,
            f"{profit_margin:.2f}",
            product.low_stock_threshold,
            'Low Stock' if product.is_low_stock else 'In Stock'
        ])
    
    # Prepare response
    output.seek(0)
    response = HttpResponse(output.read(), content_type='text/csv')
    response['Content-Disposition'] = f'attachment; filename=inventory_report_{timezone.now().date()}.csv'
    
    return response
