from .models import Notification, CompanySettings
from branches.models import Branch

def company_settings_processor(request):
    """
    Context processor for company settings
    """
    try:
        # If user is authenticated and has a branch, get branch-specific settings
        if request.user.is_authenticated and hasattr(request, 'branch') and request.branch:
            settings = CompanySettings.objects.filter(branch=request.branch).first()
            if not settings:
                # Fallback to global settings if branch-specific not found
                settings = CompanySettings.objects.filter(branch__isnull=True).first()
        else:
            # Get global settings
            settings = CompanySettings.objects.filter(branch__isnull=True).first()
    except:
        settings = None
    
    return {'company_settings': settings}

def current_branch_processor(request):
    """
    Context processor for current branch information
    """
    if not request.user.is_authenticated:
        return {'current_branch': None, 'all_branches': []}
    
    current_branch = getattr(request, 'branch', None)
    if not current_branch and hasattr(request.user, 'branch'):
        current_branch = request.user.branch
    
    # For system owners and superusers, provide a list of all branches
    if hasattr(request.user, 'is_system_owner') and request.user.is_system_owner or request.user.is_superuser:
        all_branches = Branch.objects.all().order_by('name')
    else:
        all_branches = []
    
    return {
        'current_branch': current_branch,
        'all_branches': all_branches
    }

def sidebar_items(request):
    """
    Context processor for sidebar menu items based on user role
    """
    if not request.user.is_authenticated:
        return {}
    
    # Base menu items for all roles
    menu_items = [
        {'name': 'Dashboard', 'url': 'dashboard', 'icon': 'fas fa-tachometer-alt'}
    ]
    
    # System Owner menu items
    if hasattr(request.user, 'is_system_owner') and request.user.is_system_owner:
        system_owner_menu = []
        
        # Add menu items safely
        try:
            from django.urls import reverse, NoReverseMatch
            
            # Core system owner menu items
            safe_urls = [
                {'name': 'Dashboard', 'url': 'dashboard', 'icon': 'fas fa-tachometer-alt'},
                {'name': 'Branches', 'url': 'branch_list', 'icon': 'fas fa-store'},
                {'name': 'Users', 'url': 'user_list', 'icon': 'fas fa-user-cog'},
                {'name': 'Settings', 'url': 'company_settings', 'icon': 'fas fa-cogs'}
            ]
            
            # Add items only if their URLs exist
            for item in safe_urls:
                try:
                    reverse(item['url'])
                    system_owner_menu.append(item)
                except NoReverseMatch:
                    pass
            
            # Try to add license management
            try:
                reverse('license_list')
                system_owner_menu.append({'name': 'Licenses', 'url': 'license_list', 'icon': 'fas fa-key'})
            except NoReverseMatch:
                pass
                
            # Try to add business creation
            try:
                reverse('admin_create_business')
                system_owner_menu.append({'name': 'Businesses', 'url': 'admin_create_business', 'icon': 'fas fa-building'})
            except NoReverseMatch:
                pass
                
            # Try to add audit logs
            try:
                reverse('audit_logs')
                system_owner_menu.append({'name': 'Audit Logs', 'url': 'audit_logs', 'icon': 'fas fa-history'})
            except NoReverseMatch:
                pass
        except ImportError:
            # Fallback if there's an issue with imports
            system_owner_menu = [
                {'name': 'Dashboard', 'url': 'dashboard', 'icon': 'fas fa-tachometer-alt'},
                {'name': 'Branches', 'url': 'branch_list', 'icon': 'fas fa-store'},
                {'name': 'Users', 'url': 'user_list', 'icon': 'fas fa-user-cog'}
            ]
            
        menu_items = system_owner_menu
    
    # Admin menu items
    elif request.user.role == 'admin' or request.user.is_superuser:
        admin_menu = [
            {'name': 'Branches', 'url': 'branch_list', 'icon': 'fas fa-store'},
            {'name': 'Users', 'url': 'user_list', 'icon': 'fas fa-user-cog'},
            {'name': 'Settings', 'url': 'company_settings', 'icon': 'fas fa-cogs'}
        ]
        
        # Add feature menu items only if they have corresponding url patterns
        try:
            from django.urls import reverse
            if reverse('sale_list'):
                admin_menu.insert(0, {'name': 'Sales', 'url': 'sale_list', 'icon': 'fas fa-shopping-cart'})
            if reverse('product_list'):
                admin_menu.insert(1, {'name': 'Products', 'url': 'product_list', 'icon': 'fas fa-box'})
            # Add Categories menu item for admin
            try:
                if reverse('category_list'):
                    admin_menu.insert(2, {'name': 'Categories', 'url': 'category_list', 'icon': 'fas fa-tags'})
            except:
                pass
            if reverse('customer_list'):
                admin_menu.insert(3, {'name': 'Customers', 'url': 'customer_list', 'icon': 'fas fa-users'})
            if reverse('reports'):
                admin_menu.insert(3, {'name': 'Reports', 'url': 'reports', 'icon': 'fas fa-chart-bar'})
        except:
            # If any url is not found, just skip it
            pass
            
        menu_items.extend(admin_menu)
    
    # Manager menu items
    elif request.user.role == 'manager':
        manager_menu = [
            {'name': 'Branches', 'url': 'branch_list', 'icon': 'fas fa-store'},
            {'name': 'Settings', 'url': 'company_settings', 'icon': 'fas fa-cogs'}
        ]
        
        # Add feature menu items only if they have corresponding url patterns
        try:
            from django.urls import reverse
            if reverse('sale_list'):
                manager_menu.insert(0, {'name': 'Sales', 'url': 'sale_list', 'icon': 'fas fa-shopping-cart'})
            if reverse('product_list'):
                manager_menu.insert(1, {'name': 'Products', 'url': 'product_list', 'icon': 'fas fa-box'})
            # Add Categories menu item for manager
            try:
                if reverse('category_list'):
                    manager_menu.insert(2, {'name': 'Categories', 'url': 'category_list', 'icon': 'fas fa-tags'})
            except:
                pass
            if reverse('customer_list'):
                manager_menu.insert(3, {'name': 'Customers', 'url': 'customer_list', 'icon': 'fas fa-users'})
            if reverse('reports'):
                manager_menu.insert(3, {'name': 'Reports', 'url': 'reports', 'icon': 'fas fa-chart-bar'})
        except:
            # If any url is not found, just skip it
            pass
            
        menu_items.extend(manager_menu)
    
    # Cashier menu items
    elif request.user.role == 'cashier':
        cashier_menu = []
        
        # Add feature menu items only if they have corresponding url patterns
        try:
            from django.urls import reverse
            if reverse('sale_list'):
                cashier_menu.append({'name': 'Sales', 'url': 'sale_list', 'icon': 'fas fa-shopping-cart'})
            if reverse('product_list'):
                cashier_menu.append({'name': 'Products', 'url': 'product_list', 'icon': 'fas fa-box'})
            # Categories are hidden from cashiers
            # Categories are only accessible to admins and managers
            if reverse('customer_list'):
                cashier_menu.append({'name': 'Customers', 'url': 'customer_list', 'icon': 'fas fa-users'})
            # Add direct link to cashier's own sales report
            try:
                cashier_menu.append({
                    'name': 'My Sales Report', 
                    'url': 'reports/cashiers/' + str(request.user.id) + '/', 
                    'icon': 'fas fa-chart-line',
                    'is_absolute_url': True  # Flag this as an absolute URL
                })
            except Exception as e:
                print(f"Error adding cashier report link: {str(e)}")
        except:
            # If any url is not found, just skip it
            pass
            
        menu_items.extend(cashier_menu)
    
    return {'sidebar_items': menu_items}

def notification_count(request):
    """
    Context processor for notification count
    """
    if not request.user.is_authenticated:
        return {'notification_count': 0, 'notifications': []}
    
    # System owners see all notifications
    if hasattr(request.user, 'is_system_owner') and request.user.is_system_owner:
        notifications = Notification.objects.filter(
            is_read=False
        ).order_by('-created_at')[:5]
    # Filter notifications by branch if not admin
    elif request.user.role != 'admin':
        notifications = Notification.objects.filter(
            branch=request.user.branch, 
            is_read=False
        ).order_by('-created_at')[:5]
    else:
        # Regular admins only see their branch notifications
        if hasattr(request.user, 'branch') and request.user.branch:
            notifications = Notification.objects.filter(
                branch=request.user.branch,
                is_read=False
            ).order_by('-created_at')[:5]
        else:
            notifications = Notification.objects.filter(
                is_read=False
            ).order_by('-created_at')[:5]
    
    return {
        'notification_count': notifications.count(),
        'notifications': notifications
    }
