from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required, user_passes_test
from django.contrib import messages
from django.db import transaction
from django.urls import reverse
from django.conf import settings
from django.contrib.auth import update_session_auth_hash

from .models import User
from .forms import (
    InitialAdminSetupForm, 
    UserForm, 
    ProfileUpdateForm, 
    PasswordChangeForm
)
from branches.models import Branch

def initial_setup(request):
    """
    Initial setup view to create the first admin user and company
    """
    # Check if any users exist
    if User.objects.exists():
        messages.warning(request, 'Setup has already been completed.')
        return redirect('login')
    
    if request.method == 'POST':
        form = InitialAdminSetupForm(request.POST)
        if form.is_valid():
            with transaction.atomic():
                # Create default branch for the company
                company_name = form.cleaned_data.pop('company_name')
                main_branch = Branch.objects.create(
                    name=f"{company_name} - Main Branch",
                    address="Main Address",
                    is_active=True
                )
                
                # Create admin user
                user = form.save(commit=False)
                user.role = 'admin'
                user.is_staff = True
                user.is_superuser = True
                user.branch = main_branch
                user.save()
                
                messages.success(request, 'Admin user created successfully. You can now log in.')
                return redirect('login')
    else:
        form = InitialAdminSetupForm()
    
    return render(request, 'accounts/setup.html', {'form': form})

@login_required
def profile(request):
    """
    User profile view with profile picture upload
    """
    if request.method == 'POST':
        form = ProfileUpdateForm(request.POST, request.FILES, instance=request.user)
        if form.is_valid():
            form.save()
            messages.success(request, 'Your profile has been updated successfully.')
            return redirect('profile')
    else:
        form = ProfileUpdateForm(instance=request.user)
    
    context = {
        'form': form,
        'user': request.user
    }
    return render(request, 'accounts/profile.html', context)

@login_required
def change_password(request):
    """
    Change password view
    """
    if request.method == 'POST':
        form = PasswordChangeForm(request.user, request.POST)
        if form.is_valid():
            request.user.set_password(form.cleaned_data['new_password'])
            request.user.save()
            update_session_auth_hash(request, request.user)
            messages.success(request, 'Your password has been changed successfully.')
            return redirect('profile')
    else:
        form = PasswordChangeForm(request.user)
    
    context = {
        'form': form,
    }
    return render(request, 'accounts/change_password.html', context)

def is_admin_or_system_owner(user):
    """
    Check if user is an admin or system owner
    """
    return user.is_authenticated and (user.role == 'admin' or user.role == 'system_owner' or user.is_superuser)

def is_system_owner(user):
    """
    Check if user is a system owner
    """
    return user.is_authenticated and user.role == 'system_owner'

@login_required
@user_passes_test(is_admin_or_system_owner)
def user_list(request):
    """
    List users based on user role:
    - System owners see all users
    - Admins only see users from their branch
    """
    # If system owner, show all users
    if hasattr(request.user, 'is_system_owner') and request.user.is_system_owner:
        users = User.objects.all().order_by('email')
    # Admins only see users they created in their branch
    elif request.user.role == 'admin':
        users = User.objects.filter(
            branch=request.user.branch,
            created_by=request.user
        ).exclude(role='system_owner').order_by('email')
    # Default fallback (shouldn't reach here due to decorator)
    else:
        users = User.objects.none()
    
    # Pre-calculate user counts by role for dashboard cards
    if hasattr(request.user, 'is_system_owner') and request.user.is_system_owner:
        admin_count = User.objects.filter(role='admin').count()
        manager_count = User.objects.filter(role='manager').count()
        cashier_count = User.objects.filter(role='cashier').count()
        system_owner_count = User.objects.filter(role='system_owner').count()
    else:
        # For admins, only count users they created in their branch
        admin_count = User.objects.filter(
            role='admin', 
            branch=request.user.branch,
            created_by=request.user
        ).count()
        manager_count = User.objects.filter(
            role='manager', 
            branch=request.user.branch,
            created_by=request.user
        ).count()
        cashier_count = User.objects.filter(
            role='cashier', 
            branch=request.user.branch,
            created_by=request.user
        ).count()
        system_owner_count = 0
    
    context = {
        'users': users,
        'admin_count': admin_count,
        'manager_count': manager_count,
        'cashier_count': cashier_count,
        'system_owner_count': system_owner_count
    }
    return render(request, 'accounts/user_list.html', context)

@login_required
@user_passes_test(is_admin_or_system_owner)
def user_add(request):
    """
    Add a new user (admin only)
    """
    if request.method == 'POST':
        form = UserForm(request.POST, request=request)
        if form.is_valid():
            user = form.save(commit=False)
            password = form.cleaned_data.get('password')
            
            # Generate a random password if not provided
            if not password:
                import secrets
                import string
                alphabet = string.ascii_letters + string.digits
                password = ''.join(secrets.choice(alphabet) for i in range(10))
            
            # Set created_by to the current user if not a system owner
            if not request.user.is_system_owner:
                user.created_by = request.user
                # Ensure the user is assigned to the admin's branch
                user.branch = request.user.branch
            
            user.set_password(password)
            user.save()
            
            # Store the password in session for the success page
            request.session['new_user_email'] = user.email
            request.session['new_user_password'] = password
            request.session['new_user_name'] = user.get_full_name()
            request.session['new_user_role'] = dict(User.ROLE_CHOICES).get(user.role, user.role)
            request.session['new_user_branch'] = user.branch.name if user.branch else "Not assigned"
    else:
        form = UserForm()
    
    context = {
        'form': form,
        'title': 'Add User',
    }
    return render(request, 'accounts/user_form.html', context)

@login_required
@user_passes_test(is_admin_or_system_owner)
def user_creation_success(request):
    """
    Display user creation success with credentials
    """
    context = {
        'user_email': request.session.get('new_user_email', ''),
        'user_password': request.session.get('new_user_password', ''),
        'user_name': request.session.get('new_user_name', ''),
        'user_role': request.session.get('new_user_role', ''),
        'user_branch': request.session.get('new_user_branch', ''),
    }
    
    # Clear session data for security
    for key in ['new_user_email', 'new_user_password', 'new_user_name', 'new_user_role', 'new_user_branch', 'is_password_reset']:
        if key in request.session:
            del request.session[key]
    
    return render(request, 'accounts/user_creation_success.html', context)

@login_required
@user_passes_test(is_admin_or_system_owner)
def user_edit(request, user_id):
    """
    Edit an existing user
    - System owners can edit any user
    - Admins can only edit users from their branch
    """
    user = get_object_or_404(User, id=user_id)
    
    # Check if admin is trying to edit a user from a different branch
    if request.user.role == 'admin' and user.branch != request.user.branch:
        messages.error(request, 'You can only edit users from your branch.')
        return redirect('user_list')
    
    # Check if admin is trying to edit a user from a different branch or not created by them
    if request.user.role == 'admin':
        if user.branch != request.user.branch or user.created_by != request.user:
            messages.error(request, 'You can only edit users you created in your branch.')
            return redirect('user_list')
    
    if request.method == 'POST':
        form = UserForm(request.POST, instance=user, request=request)
        if form.is_valid():
            # Prevent admins from changing user roles or branches
            if request.user.role == 'admin':
                if form.cleaned_data['role'] != user.role:
                    messages.error(request, 'You cannot change user roles.')
                    return redirect('user_edit', user_id=user.id)
                if form.cleaned_data['branch'] != user.branch:
                    messages.error(request, 'You cannot move users between branches.')
                    return redirect('user_edit', user_id=user.id)
            
            form.save()
            messages.success(request, f'User {user.email} has been updated.')
            return redirect('user_list')
    else:
        form = UserForm(instance=user, request=request)
    
    context = {
        'form': form,
        'title': 'Edit User',
        'user_obj': user,
    }
    return render(request, 'accounts/user_form.html', context)

@login_required
@user_passes_test(is_admin_or_system_owner)
def user_reset_password(request, user_id):
    """
    Reset password for a user (admin only)
    """
    user = get_object_or_404(User, id=user_id)
    
    # Prevent resetting own password through this method
    if user == request.user:
        messages.warning(request, 'You cannot reset your own password this way. Please use the change password option in your profile.')
        return redirect('user_list')
    
    # Generate new random password
    import secrets
    import string
    alphabet = string.ascii_letters + string.digits
    new_password = ''.join(secrets.choice(alphabet) for i in range(10))
    
    # Set the new password
    user.set_password(new_password)
    user.save()
    
    # Store details in session for the success page
    request.session['new_user_email'] = user.email
    request.session['new_user_password'] = new_password
    request.session['new_user_name'] = user.get_full_name()
    request.session['new_user_role'] = dict(User.ROLE_CHOICES).get(user.role, user.role)
    request.session['new_user_branch'] = user.branch.name if user.branch else "Not assigned"
    request.session['is_password_reset'] = True
    
    return redirect('user_creation_success')

@login_required
@user_passes_test(is_system_owner)  # Only system owners can delete users
def user_delete(request, user_id):
    """
    Delete a user (system owner only)
    """
    user = get_object_or_404(User, id=user_id)
    
    # Prevent self-deletion
    if user == request.user:
        messages.error(request, 'You cannot delete your own account.')
        return redirect('user_list')
        
    # Prevent deletion of system owners
    if user.role == 'system_owner':
        messages.error(request, 'System owner accounts cannot be deleted.')
        return redirect('user_list')
    
    if request.method == 'POST':
        user_email = user.email
        user.delete()
        messages.success(request, f'User {user_email} has been deleted.')
        return redirect('user_list')
    
    context = {
        'user_obj': user,
    }
    return render(request, 'accounts/user_confirm_delete.html', context)
