from django.db import models
from django.conf import settings

class CompanySettings(models.Model):
    """
    Company settings model for storing company details
    """
    company_name = models.CharField(max_length=100)
    company_address = models.TextField()
    contact_email = models.EmailField()
    contact_phone = models.CharField(max_length=20)
    logo = models.ImageField(upload_to='company_logos/', null=True, blank=True)
    tax_rate = models.DecimalField(max_digits=5, decimal_places=2, default=0.0)
    currency_symbol = models.CharField(max_length=5, default='KSh')
    receipt_footer = models.TextField(blank=True)
    branch = models.ForeignKey('branches.Branch', on_delete=models.SET_NULL, null=True, blank=True)
    
    class Meta:
        verbose_name = 'Company Settings'
        verbose_name_plural = 'Company Settings'
    
    def __str__(self):
        return self.company_name
    
    @classmethod
    def get_settings(cls):
        """
        Get or create company settings
        """
        settings, created = cls.objects.get_or_create(
            pk=1,
            defaults={
                'company_name': 'AsapLoads POS',
                'company_address': 'Main Address',
                'contact_email': 'info@asaploadspos.com',
                'contact_phone': '123-456-7890',
                'currency_symbol': 'KSh',
                'receipt_footer': 'Thank you for your business!'
            }
        )
        return settings

class Notification(models.Model):
    """
    Notification model for alerts like low stock
    """
    title = models.CharField(max_length=100)
    message = models.TextField()
    created_at = models.DateTimeField(auto_now_add=True)
    is_read = models.BooleanField(default=False)
    branch = models.ForeignKey('branches.Branch', on_delete=models.CASCADE, null=True, blank=True)
    
    class Meta:
        ordering = ['-created_at']
    
    def __str__(self):
        return self.title
    
    @classmethod
    def create_low_stock_notification(cls, product, branch=None):
        """
        Create a notification for low stock
        """
        title = f"Low Stock Alert: {product.name}"
        message = f"Product '{product.name}' is running low on stock. Current quantity: {product.quantity}."
        return cls.objects.create(title=title, message=message, branch=branch)
        
    @classmethod
    def create_sale_cancellation_notification(cls, sale, user, branch=None):
        """
        Create a notification for sale cancellation (suspicious activity)
        """
        title = f"Sale Cancelled: #{sale.invoice_number}"
        message = f"Sale #{sale.invoice_number} has been cancelled by {user.get_full_name()} ({user.role}). Total amount: {sale.total_amount}. This may require your attention."
        return cls.objects.create(title=title, message=message, branch=branch)

class AuditLog(models.Model):
    """
    Audit log for tracking important actions
    """
    ACTION_CHOICES = (
        ('create', 'Create'),
        ('update', 'Update'),
        ('delete', 'Delete'),
        ('login', 'Login'),
        ('logout', 'Logout'),
        ('license', 'License'),
        ('other', 'Other'),
    )
    
    user = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.SET_NULL, null=True, blank=True)
    action = models.CharField(max_length=10, choices=ACTION_CHOICES)
    entity_type = models.CharField(max_length=50)  # e.g., 'user', 'product', 'sale'
    entity_id = models.IntegerField(null=True, blank=True)
    details = models.TextField(blank=True)
    ip_address = models.GenericIPAddressField(null=True, blank=True)
    timestamp = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        ordering = ['-timestamp']
    
    def __str__(self):
        return f"{self.action} {self.entity_type} at {self.timestamp}"
    
    @classmethod
    def log_action(cls, user, action, entity_type, entity_id=None, details='', ip_address=None):
        """
        Create an audit log entry
        """
        return cls.objects.create(
            user=user,
            action=action,
            entity_type=entity_type,
            entity_id=entity_id,
            details=details,
            ip_address=ip_address
        )
