from django import forms
from django.contrib.auth.forms import UserCreationForm, AuthenticationForm
from django.core.exceptions import ValidationError
from .models import User

class CustomAuthenticationForm(AuthenticationForm):
    """
    Custom authentication form with styled fields
    """
    username = forms.EmailField(
        widget=forms.EmailInput(attrs={'class': 'form-control', 'placeholder': 'Email'}),
        label="Email"
    )
    password = forms.CharField(
        widget=forms.PasswordInput(attrs={'class': 'form-control', 'placeholder': 'Password'}),
        label="Password"
    )
    
    def confirm_login_allowed(self, user):
        if not user.is_active:
            raise ValidationError(
                "This account is inactive.",
                code='inactive',
            )
        
        # Superusers bypass license checks
        if user.is_superuser:
            return
        
        # Check if license is valid for this user's branch
        if user.branch and not user.branch.has_valid_license():
            raise ValidationError(
                "Your branch license has expired or is invalid. Please contact the administrator.",
                code='license_invalid',
            )
        
        super().confirm_login_allowed(user)

class InitialAdminSetupForm(UserCreationForm):
    """
    Form for initial admin user setup
    """
    company_name = forms.CharField(
        max_length=100,
        widget=forms.TextInput(attrs={'class': 'form-control', 'placeholder': 'Company Name'})
    )
    
    class Meta:
        model = User
        fields = ('email', 'first_name', 'last_name', 'password1', 'password2', 'company_name')
        widgets = {
            'email': forms.EmailInput(attrs={'class': 'form-control', 'placeholder': 'Email'}),
            'first_name': forms.TextInput(attrs={'class': 'form-control', 'placeholder': 'First Name'}),
            'last_name': forms.TextInput(attrs={'class': 'form-control', 'placeholder': 'Last Name'}),
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['password1'].widget.attrs.update({'class': 'form-control', 'placeholder': 'Password'})
        self.fields['password2'].widget.attrs.update({'class': 'form-control', 'placeholder': 'Confirm Password'})

class UserForm(forms.ModelForm):
    """
    Form for creating or updating a user
    """
    password = forms.CharField(
        widget=forms.PasswordInput(attrs={'class': 'form-control', 'placeholder': 'Temporary Password'}),
        required=False,
        help_text="For new users: Enter a temporary password or leave empty to generate one automatically. For existing users: Leave empty to keep the current password."
    )
    
    class Meta:
        model = User
        fields = ('email', 'first_name', 'last_name', 'role', 'branch', 'is_active')
        widgets = {
            'email': forms.EmailInput(attrs={'class': 'form-control', 'placeholder': 'Email'}),
            'first_name': forms.TextInput(attrs={'class': 'form-control', 'placeholder': 'First Name'}),
            'last_name': forms.TextInput(attrs={'class': 'form-control', 'placeholder': 'Last Name'}),
            'role': forms.Select(attrs={'class': 'form-control'}),
            'branch': forms.Select(attrs={'class': 'form-control'}),
            'is_active': forms.CheckboxInput(attrs={'class': 'form-check-input'}),
        }
    
    def __init__(self, *args, **kwargs):
        self.request = kwargs.pop('request', None)
        super().__init__(*args, **kwargs)
        
        # Set password field as not required for existing users
        if self.instance and self.instance.pk:
            self.fields['password'].required = False
            self.fields['password'].help_text = "Leave blank to keep the current password."
        else:
            self.fields['password'].help_text = "Leave blank to generate a random password."
        
        # Handle role and branch restrictions based on user type
        if self.request and not self.request.user.is_superuser:
            # Remove system_owner from role choices for non-superusers
            role_choices = [
                (role, label) for role, label in self.fields['role'].choices 
                if role != 'system_owner'
            ]
            
            # If admin, only allow creating cashier and manager roles
            if self.request.user.role == 'admin':
                role_choices = [
                    (role, label) for role, label in role_choices
                    if role in ['cashier', 'manager']
                ]
                
                # Set the branch to admin's branch and make it read-only
                self.fields['branch'].initial = self.request.user.branch
                self.fields['branch'].disabled = True
                self.fields['branch'].widget.attrs['readonly'] = True
            
            self.fields['role'].choices = role_choices
    
    def save(self, commit=True):
        user = super().save(commit=False)
        password = self.cleaned_data.get('password')
        if password:
            user.set_password(password)
        
        # Set created_by to the current user for new users
        if not user.pk and hasattr(self, 'request') and self.request.user.is_authenticated:
            user.created_by = self.request.user
        
        if commit:
            user.save()
        return user

class ProfileUpdateForm(forms.ModelForm):
    """Form for updating user profile"""
    profile_picture = forms.ImageField(
        required=False,
        widget=forms.FileInput(attrs={'class': 'form-control', 'accept': 'image/*'})
    )
    
    class Meta:
        model = User
        fields = ('first_name', 'last_name', 'email', 'profile_picture')
        widgets = {
            'first_name': forms.TextInput(attrs={'class': 'form-control'}),
            'last_name': forms.TextInput(attrs={'class': 'form-control'}),
            'email': forms.EmailInput(attrs={'class': 'form-control'}),
        }

class PasswordChangeForm(forms.Form):
    """
    Form for changing user password
    """
    current_password = forms.CharField(
        widget=forms.PasswordInput(attrs={'class': 'form-control', 'placeholder': 'Current Password'})
    )
    new_password = forms.CharField(
        widget=forms.PasswordInput(attrs={'class': 'form-control', 'placeholder': 'New Password'})
    )
    confirm_password = forms.CharField(
        widget=forms.PasswordInput(attrs={'class': 'form-control', 'placeholder': 'Confirm New Password'})
    )
    
    def __init__(self, user, *args, **kwargs):
        self.user = user
        super().__init__(*args, **kwargs)
    
    def clean_current_password(self):
        current_password = self.cleaned_data.get('current_password')
        if not self.user.check_password(current_password):
            raise forms.ValidationError("Your current password is incorrect.")
        return current_password
    
    def clean(self):
        cleaned_data = super().clean()
        new_password = cleaned_data.get('new_password')
        confirm_password = cleaned_data.get('confirm_password')
        
        if new_password and confirm_password:
            if new_password != confirm_password:
                self.add_error('confirm_password', "The new passwords don't match.")
        
        return cleaned_data
