from django import forms
from django.forms import ModelForm, inlineformset_factory
from django.utils.translation import gettext_lazy as _
from django.core.validators import RegexValidator, MinValueValidator, FileExtensionValidator
import os

from crispy_forms.helper import FormHelper
from crispy_forms.layout import Layout, Row, Column, Field, Fieldset, ButtonHolder, Submit, Div

from .models import (
    Supplier, SupplierCategory, SupplierContact, 
    SupplierDocument, SupplierNote
)

class SupplierCategoryForm(forms.ModelForm):
    class Meta:
        model = SupplierCategory
        fields = ['name', 'description', 'is_active']
        widgets = {
            'name': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Enter category name')
            }),
            'description': forms.Textarea(attrs={
                'class': 'form-control',
                'rows': 3,
                'placeholder': _('Enter a brief description')
            }),
            'is_active': forms.CheckboxInput(attrs={
                'class': 'form-check-input'
            })
        }
        labels = {
            'name': _('Category Name'),
            'is_active': _('Active')
        }

class SupplierForm(forms.ModelForm):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.helper = FormHelper()
        self.helper.form_method = 'post'
        self.helper.form_id = 'supplier-form'
        self.helper.form_class = 'form-horizontal'
        self.helper.label_class = 'form-label'
        self.helper.field_class = 'w-100'
        
        # Add form actions
        self.helper.layout = Layout(
            Fieldset(
                'Basic Information',
                Row(
                    Column('name', css_class='col-md-6'),
                    Column('company_name', css_class='col-md-6'),
                    css_class='mb-3'
                ),
                Row(
                    Column('email', css_class='col-md-6'),
                    Column('phone', css_class='col-md-6'),
                    css_class='mb-3'
                ),
                Row(
                    Column('tax_identification_number', css_class='col-md-6'),
                    css_class='mb-3'
                ),
                Row(
                    Column('address_line1', css_class='col-md-12'),
                    css_class='mb-3'
                ),
                css_class='mb-4'
            ),
            Fieldset(
                'Payment Information',
                Row(
                    Column('payment_terms', css_class='col-md-6'),
                    Column('credit_limit', css_class='col-md-6'),
                    css_class='mb-3'
                ),
                'notes',
                css_class='mb-4'
            ),
            Div(
                Submit('submit', 'Save Supplier', css_class='btn btn-primary me-2'),
                css_class='d-flex justify-content-end mt-4'
            )
        )

    payment_terms = forms.IntegerField(
        required=True,
        min_value=0,
        initial=30,
        widget=forms.NumberInput(attrs={
            'class': 'form-control',
            'placeholder': 'Enter payment terms in days',
            'min': '0',
            'step': '1'
        }),
        help_text='Payment terms in days (e.g., 30 for Net 30)'
    )
    
    class Meta:
        model = Supplier
        fields = [
            'name', 'company_name', 'email', 'phone',
            'tax_identification_number', 'address_line1',
            'payment_terms', 'credit_limit', 'notes'
        ]
        widgets = {
            'name': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Supplier name')
            }),
            'supplier_type': forms.Select(attrs={
                'class': 'form-select',
            }),
            'company_name': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Company name')
            }),
            'branches': forms.SelectMultiple(attrs={
                'class': 'form-select',
                'data-choices': 'data-choices',
                'data-options': '{"removeItemButton": true, "maxItemCount": 10}',
                'multiple': 'multiple'
            }),
            'email': forms.EmailInput(attrs={
                'class': 'form-control',
                'placeholder': 'example@domain.com'
            }),
            'phone': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': '+1234567890',
                'data-phone-input': ''
            }),
            'mobile': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': '+1234567890',
                'data-phone-input': ''
            }),
            'website': forms.URLInput(attrs={
                'class': 'form-control',
                'placeholder': 'https://example.com'
            }),
            'address_line1': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Street address, P.O. box, company name')
            }),
            'address_line2': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Apartment, suite, unit, building, floor, etc.')
            }),
            'city': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('City')
            }),
            'state': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('State / Province / Region')
            }),
            'postal_code': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Postal / Zip code')
            }),
            'tax_identification_number': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Tax ID')
            }),
            'registration_number': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Business registration number')
            }),
            'vat_number': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('VAT number')
            }),
            'currency': forms.Select(attrs={
                'class': 'form-select',
            }),
            'payment_terms': forms.NumberInput(attrs={
                'class': 'form-control',
                'placeholder': '30',
                'min': '0',
                'step': '1'
            }),
            'credit_limit': forms.NumberInput(attrs={
                'class': 'form-control',
                'placeholder': '0.00',
                'min': '0',
                'step': '0.01'
            }),
            'is_active': forms.CheckboxInput(attrs={
                'class': 'form-check-input',
            }),
            'preferred': forms.CheckboxInput(attrs={
                'class': 'form-check-input',
            }),
            'notes': forms.Textarea(attrs={
                'class': 'form-control',
                'rows': 3,
                'placeholder': _('Any additional notes or comments')
            }),
        }
        help_texts = {
            'payment_terms': _('Number of days for payment (e.g., 30 for Net 30)'),
            'credit_limit': _('Maximum credit amount allowed for this supplier'),
            'preferred': _('Mark as a preferred supplier')
        }

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        # Set empty label for optional fields
        if 'category' in self.fields:
            self.fields['category'].empty_label = _('Select a category')
        if 'currency' in self.fields:
            self.fields['currency'].empty_label = _('Select a currency')
        
        # Add form-control class to all fields by default
        for field_name, field in self.fields.items():
            if 'class' not in field.widget.attrs:
                field.widget.attrs['class'] = 'form-control'

class SupplierContactForm(forms.ModelForm):
    class Meta:
        model = SupplierContact
        fields = [
            'first_name', 'last_name', 'position', 'contact_type',
            'email', 'phone', 'mobile', 'is_primary', 'notes'
        ]
        widgets = {
            'first_name': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('First name')
            }),
            'last_name': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Last name')
            }),
            'position': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Job title/position')
            }),
            'contact_type': forms.Select(attrs={
                'class': 'form-select',
            }),
            'email': forms.EmailInput(attrs={
                'class': 'form-control',
                'placeholder': 'example@domain.com'
            }),
            'phone': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': '+1234567890',
                'data-phone-input': ''
            }),
            'mobile': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': '+1234567890',
                'data-phone-input': ''
            }),
            'department': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Department name')
            }),
            'is_primary': forms.CheckboxInput(attrs={
                'class': 'form-check-input',
            }),
            'notes': forms.Textarea(attrs={
                'class': 'form-control',
                'rows': 2,
                'placeholder': _('Any additional notes about this contact')
            }),
        }
        help_texts = {
            'is_primary': _('Set as the primary contact for this supplier'),
        }

class SupplierDocumentForm(forms.ModelForm):
    class Meta:
        model = SupplierDocument
        fields = ['document_type', 'title', 'file', 'valid_from', 'valid_to', 'description']
        widgets = {
            'document_type': forms.Select(attrs={
                'class': 'form-select',
            }),
            'title': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Document title')
            }),
            'file': forms.FileInput(attrs={
                'class': 'form-control',
                'accept': '.pdf,.doc,.docx,.xls,.xlsx,.jpg,.jpeg,.png',
            }),
            'valid_from': forms.DateInput(attrs={
                'class': 'form-control',
                'data-provider': 'flatpickr',
                'data-date-format': 'Y-m-d',
                'placeholder': 'YYYY-MM-DD'
            }),
            'valid_to': forms.DateInput(attrs={
                'class': 'form-control',
                'data-provider': 'flatpickr',
                'data-date-format': 'Y-m-d',
                'placeholder': 'YYYY-MM-DD'
            }),
            'description': forms.Textarea(attrs={
                'class': 'form-control',
                'rows': 2,
                'placeholder': _('Document description or notes')
            }),
        }
        help_texts = {
            'file': _('Supported formats: PDF, DOC, DOCX, XLS, XLSX, JPG, PNG (Max 10MB)'),
        }
    
    def clean_file(self):
        file = self.cleaned_data.get('file')
        if file:
            # Check file size (10MB max)
            max_size = 10 * 1024 * 1024  # 10MB
            if file.size > max_size:
                raise forms.ValidationError(_('File size must be no more than 10MB.'))
            
            # Check file extension
            valid_extensions = ['.pdf', '.doc', '.docx', '.xls', '.xlsx', '.jpg', '.jpeg', '.png']
            ext = os.path.splitext(file.name)[1].lower()
            if ext not in valid_extensions:
                raise forms.ValidationError(_('Unsupported file type. Supported formats: PDF, DOC, DOCX, XLS, XLSX, JPG, PNG'))
        
        return file

class SupplierNoteForm(forms.ModelForm):
    class Meta:
        model = SupplierNote
        fields = ['note']
        widgets = {
            'note': forms.Textarea(attrs={
                'class': 'form-control',
                'rows': 3,
                'placeholder': _('Add a note about this supplier...')
            }),
        }
        labels = {
            'note': _('Add Note')
        }
