from django import forms
from django.forms import inlineformset_factory, BaseInlineFormSet
from django.utils import timezone

from .models import PurchaseOrder, PurchaseOrderItem, GoodsReceipt, GoodsReceiptItem
from suppliers.models import Supplier

class PurchaseOrderForm(forms.ModelForm):
    class Meta:
        model = PurchaseOrder
        fields = ['supplier', 'order_date', 'expected_delivery_date', 'notes']
        widgets = {
            'order_date': forms.DateInput(attrs={'type': 'date'}, format='%Y-%m-%d'),
            'expected_delivery_date': forms.DateInput(attrs={'type': 'date'}, format='%Y-%m-%d'),
            'notes': forms.Textarea(attrs={'rows': 3}),
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['supplier'].queryset = Supplier.objects.filter(is_active=True)
        self.fields['order_date'].initial = timezone.now().date()

class PurchaseOrderItemForm(forms.ModelForm):
    class Meta:
        model = PurchaseOrderItem
        fields = ['product', 'quantity', 'unit_price']
        widgets = {
            'quantity': forms.NumberInput(attrs={'class': 'form-control quantity', 'min': '0.01', 'step': '0.01'}),
            'unit_price': forms.NumberInput(attrs={'class': 'form-control price', 'min': '0.01', 'step': '0.01'}),
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['product'].widget.attrs.update({'class': 'form-control product-select'})
        self.fields['quantity'].widget.attrs.update({'class': 'form-control quantity'})
        self.fields['unit_price'].widget.attrs.update({'class': 'form-control price'})

class BasePurchaseOrderItemFormSet(BaseInlineFormSet):
    def clean(self):
        super().clean()
        for form in self.forms:
            if not hasattr(form, 'cleaned_data'):
                continue
            
            data = form.cleaned_data
            if data.get('DELETE') and not data.get('id'):
                form.cleaned_data = {}
                continue
                
            if not data.get('product'):
                form.cleaned_data = {}
                continue
                
            if data.get('quantity') <= 0:
                form.add_error('quantity', 'Quantity must be greater than zero')
            
            if data.get('unit_price') <= 0:
                form.add_error('unit_price', 'Unit price must be greater than zero')

PurchaseOrderItemFormSet = inlineformset_factory(
    PurchaseOrder, 
    PurchaseOrderItem, 
    form=PurchaseOrderItemForm,
    formset=BasePurchaseOrderItemFormSet,
    extra=1,
    can_delete=True,
    min_num=1,
    validate_min=True
)

class GoodsReceiptForm(forms.ModelForm):
    class Meta:
        model = GoodsReceipt
        fields = ['received_date', 'notes']
        widgets = {
            'received_date': forms.DateInput(attrs={'type': 'date'}, format='%Y-%m-%d'),
            'notes': forms.Textarea(attrs={'rows': 3}),
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['received_date'].initial = timezone.now().date()

class GoodsReceiptItemForm(forms.ModelForm):
    class Meta:
        model = GoodsReceiptItem
        fields = ['purchase_order_item', 'quantity_received', 'batch_number', 'expiry_date']
        widgets = {
            'quantity_received': forms.NumberInput(attrs={'class': 'form-control quantity-received', 'min': '0.01', 'step': '0.01'}),
            'batch_number': forms.TextInput(attrs={'class': 'form-control'}),
            'expiry_date': forms.DateInput(attrs={'type': 'date'}, format='%Y-%m-%d'),
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['quantity_received'].widget.attrs.update({'class': 'form-control quantity-received'})
        self.fields['batch_number'].widget.attrs.update({'class': 'form-control'})
        self.fields['expiry_date'].widget.attrs.update({'class': 'form-control'})

class BaseGoodsReceiptItemFormSet(BaseInlineFormSet):
    def __init__(self, *args, **kwargs):
        self.purchase_order = kwargs.pop('purchase_order', None)
        super().__init__(*args, **kwargs)
        
        # Only show items that haven't been fully received
        if self.purchase_order:
            self.queryset = PurchaseOrderItem.objects.filter(
                purchase_order=self.purchase_order,
                quantity__gt=0
            )
            
            for form in self.forms:
                item = form.instance.purchase_order_item if form.instance else None
                if item:
                    max_qty = item.quantity - item.received_quantity
                    form.fields['quantity_received'].widget.attrs.update({
                        'max': max_qty,
                        'placeholder': f'Max: {max_qty}'
                    })
    
    def clean(self):
        super().clean()
        for form in self.forms:
            if not hasattr(form, 'cleaned_data'):
                continue
                
            data = form.cleaned_data
            if not data.get('purchase_order_item'):
                continue
                
            item = data['purchase_order_item']
            quantity_received = data.get('quantity_received', 0)
            remaining_qty = item.quantity - item.received_quantity
            
            if quantity_received <= 0:
                form.add_error('quantity_received', 'Quantity must be greater than zero')
            elif quantity_received > remaining_qty:
                form.add_error(
                    'quantity_received', 
                    f'Cannot receive more than {remaining_qty} units (ordered: {item.quantity}, received: {item.received_quantity})'
                )

def get_goods_receipt_item_formset(purchase_order, data=None, **kwargs):
    """Helper function to create a formset with the purchase order context."""
    # Extract prefix from kwargs if it exists
    prefix = kwargs.pop('prefix', 'items')
    
    formset_class = inlineformset_factory(
        GoodsReceipt,
        GoodsReceiptItem,
        form=GoodsReceiptItemForm,
        formset=BaseGoodsReceiptItemFormSet,
        extra=0,
        can_delete=False
    )
    
    # Create formset instance with the purchase order and prefix
    formset = formset_class(
        data,
        prefix=prefix,
        **kwargs
    )
    
    # Pass the purchase order to the formset
    formset.purchase_order = purchase_order
    return formset
