from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required, user_passes_test
from django.contrib import messages
from django.utils import timezone
from django.urls import reverse
from datetime import timedelta
import string
import secrets

from .models import License
from .forms import LicenseForm, LicenseActivationForm
from branches.models import Branch
from core.models import AuditLog, CompanySettings
from accounts.models import User
from core.utils import send_business_credentials_email

def is_system_owner(user):
    """
    Check if user is a system owner
    """
    return user.is_authenticated and user.role == 'system_owner'

@login_required
@user_passes_test(is_system_owner)
def license_list(request):
    """
    List all licenses (admin only)
    """
    licenses = License.objects.all().select_related('branch').order_by('-created_at')
    
    context = {
        'licenses': licenses,
    }
    return render(request, 'licenses/license_list.html', context)

@login_required
@user_passes_test(is_system_owner)
def license_add(request):
    """
    Add a new license (admin only)
    """
    if request.method == 'POST':
        form = LicenseForm(request.POST)
        if form.is_valid():
            branch = form.cleaned_data['branch']
            days = form.cleaned_data['duration']
            
            # Generate license key
            license_data = License.generate_license_key(branch.id, days=days)
            
            # Create license
            license = License.objects.create(
                license_key=license_data['license_key'],
                branch=branch,
                issue_date=license_data['issue_date'],
                expiry_date=license_data['expiry_date'],
                is_active=True
            )
            
            # Log the action
            AuditLog.log_action(
                user=request.user,
                action='license',
                entity_type='license',
                entity_id=license.id,
                details=f"Generated license for branch: {branch.name}, valid for {days} days",
                ip_address=request.META.get('REMOTE_ADDR')
            )
            
            messages.success(request, f'License key generated successfully for {branch.name}.')
            return redirect('license_list')
    else:
        form = LicenseForm()
    
    context = {
        'form': form,
        'title': 'Generate License Key',
    }
    return render(request, 'licenses/license_form.html', context)

@login_required
@user_passes_test(is_system_owner)
def license_revoke(request, license_id):
    """
    Revoke a license (admin only)
    """
    license = get_object_or_404(License, id=license_id)
    
    if request.method == 'POST':
        license.is_active = False
        license.save()
        
        # Log the action
        AuditLog.log_action(
            user=request.user,
            action='license',
            entity_type='license',
            entity_id=license.id,
            details=f"Revoked license for branch: {license.branch.name}",
            ip_address=request.META.get('REMOTE_ADDR')
        )
        
        messages.success(request, f'License for {license.branch.name} has been revoked.')
        return redirect('license_list')
    
    context = {
        'license': license,
    }
    return render(request, 'licenses/license_confirm_revoke.html', context)

@login_required
@user_passes_test(is_system_owner)
def license_renew(request, license_id):
    """
    Renew a license (admin only)
    """
    license = get_object_or_404(License, id=license_id)
    
    if request.method == 'POST':
        days = int(request.POST.get('days', 365))
        
        # Generate new license key
        license_data = License.generate_license_key(license.branch.id, days=days)
        
        # Create new license
        new_license = License.objects.create(
            license_key=license_data['license_key'],
            branch=license.branch,
            issue_date=license_data['issue_date'],
            expiry_date=license_data['expiry_date'],
            is_active=True
        )
        
        # Deactivate old license
        license.is_active = False
        license.save()
        
        # Log the action
        AuditLog.log_action(
            user=request.user,
            action='license',
            entity_type='license',
            entity_id=new_license.id,
            details=f"Renewed license for branch: {license.branch.name}, valid for {days} days",
            ip_address=request.META.get('REMOTE_ADDR')
        )
        
        messages.success(request, f'License for {license.branch.name} has been renewed for {days} days.')
        return redirect('license_list')
    
    context = {
        'license': license,
    }
    return render(request, 'licenses/license_renew.html', context)


@login_required
def license_activate(request):
    """
    License activation for business admins
    """
    # If user is already authenticated and has a branch, or is an admin/superuser, redirect to dashboard
    if request.user.is_authenticated:
        if request.user.branch is not None or request.user.role == 'admin' or request.user.is_superuser:
            return redirect('dashboard')  # or your dashboard URL name
    
    if request.method == 'POST':
        form = LicenseActivationForm(request.POST)
        if form.is_valid():
            license_key = form.cleaned_data['license_key']
            company_name = form.cleaned_data['company_name']
            address = form.cleaned_data['address']
            contact_email = form.cleaned_data['contact_email']
            contact_phone = form.cleaned_data['contact_phone']
            
            # Check if license key exists and is active
            try:
                license = License.objects.get(license_key=license_key, is_active=True)
                
                # Verify license has not expired
                if license.is_expired:
                    messages.error(request, 'This license key has expired. Please contact the system administrator.')
                    return redirect('license_activate')
                
                # Update branch information
                branch = license.branch
                branch.name = company_name
                branch.address = address
                branch.email = contact_email
                branch.phone = contact_phone
                branch.save()
                
                # Assign branch to user
                request.user.branch = branch
                request.user.role = 'admin'  # Promote to branch admin
                request.user.save()
                
                # Log the action
                AuditLog.log_action(
                    user=request.user,
                    action='license',
                    entity_type='license',
                    entity_id=license.id,
                    details=f"License activated for branch: {branch.name}",
                    ip_address=request.META.get('REMOTE_ADDR')
                )
                
                messages.success(request, f'License activated successfully! You are now the administrator for {branch.name}.')
                return redirect('dashboard')
                
            except License.DoesNotExist:
                messages.error(request, 'Invalid license key. Please check the key and try again.')
                return redirect('license_activate')
    else:
        form = LicenseActivationForm()
    
    context = {
        'form': form,
    }
    return render(request, 'licenses/license_activate.html', context)


def check_license_required(view_func):
    """
    Decorator to check if user needs to activate a license
    """
    def wrapper(request, *args, **kwargs):
        # Superusers and admins bypass all license checks
        if request.user.is_authenticated and (request.user.is_superuser or request.user.role == 'admin'):
            return view_func(request, *args, **kwargs)
            
        if request.user.is_authenticated and request.user.branch is None and request.user.role != 'admin':
            # User is authenticated but doesn't have a branch and is not an admin
            # This is likely a user who needs to activate a license
            return redirect('license_activate')
        return view_func(request, *args, **kwargs)
    return wrapper


@login_required
@user_passes_test(lambda u: u.is_superuser)
def admin_create_business(request):
    """
    Create a new business with branch, license, and admin (super admin only)
    """
    if request.method == 'POST':
        # Get form data
        branch_name = request.POST.get('branch_name')
        branch_address = request.POST.get('branch_address')
        branch_email = request.POST.get('branch_email')
        branch_phone = request.POST.get('branch_phone')
        license_duration = int(request.POST.get('license_duration', 365))
        
        # Admin account details
        first_name = request.POST.get('first_name')
        last_name = request.POST.get('last_name')
        email = request.POST.get('email')
        generate_password = request.POST.get('generate_password') == 'on'
        
        # Check if email is already in use (case-insensitive)
        if User.objects.filter(email__iexact=email).exists():
            messages.error(request, f'A user with email {email} already exists. Please use a different email.')
            return redirect('admin_create_business')
        
        # Create branch
        branch = Branch.objects.create(
            name=branch_name,
            address=branch_address,
            email=branch_email,
            phone=branch_phone,
            is_active=True
        )
        
        # Generate license key
        license_data = License.generate_license_key(branch.id, days=license_duration)
        
        # Create license
        license = License.objects.create(
            license_key=license_data['license_key'],
            branch=branch,
            issue_date=license_data['issue_date'],
            expiry_date=license_data['expiry_date'],
            is_active=True
        )
        
        # Create admin password
        if generate_password:
            # Generate random password
            alphabet = string.ascii_letters + string.digits
            password = ''.join(secrets.choice(alphabet) for i in range(10))
        else:
            password = request.POST.get('password')
        
        # Create admin user
        admin_user = User.objects.create_user(
            email=email,
            password=password,
            first_name=first_name,
            last_name=last_name,
            role='admin'
            # Don't assign branch yet - this will happen during license activation
        )
        
        # Log the actions
        AuditLog.log_action(
            user=request.user,
            action='create',
            entity_type='branch',
            entity_id=branch.id,
            details=f"Created business branch: {branch.name}",
            ip_address=request.META.get('REMOTE_ADDR')
        )
        
        AuditLog.log_action(
            user=request.user,
            action='license',
            entity_type='license',
            entity_id=license.id,
            details=f"Generated license for branch: {branch.name}, valid for {license_duration} days",
            ip_address=request.META.get('REMOTE_ADDR')
        )
        
        AuditLog.log_action(
            user=request.user,
            action='create',
            entity_type='user',
            entity_id=admin_user.id,
            details=f"Created admin user: {admin_user.email} for branch: {branch.name}",
            ip_address=request.META.get('REMOTE_ADDR')
        )
        
        # Store business and credentials in session for success page
        request.session['branch_name'] = branch_name
        request.session['branch_address'] = branch_address
        request.session['branch_email'] = branch_email
        request.session['branch_phone'] = branch_phone
        request.session['license_key'] = license_data['license_key']
        request.session['issue_date'] = license_data['issue_date'].strftime('%Y-%m-%d')
        request.session['expiry_date'] = license_data['expiry_date'].strftime('%Y-%m-%d')
        request.session['duration'] = license_duration
        request.session['admin_name'] = f"{first_name} {last_name}"
        request.session['admin_email'] = email
        request.session['admin_password'] = password
        request.session['site_url'] = request.build_absolute_uri(reverse('login'))
        
        # Send email with credentials
        email_context = {
            'branch_name': branch_name,
            'branch_address': branch_address,
            'branch_email': branch_email,
            'branch_phone': branch_phone,
            'license_key': license_data['license_key'],
            'issue_date': license_data['issue_date'].strftime('%Y-%m-%d'),
            'expiry_date': license_data['expiry_date'].strftime('%Y-%m-%d'),
            'duration': license_duration,
            'admin_name': f"{first_name} {last_name}",
            'admin_email': email,
            'admin_password': password,
            'site_url': request.build_absolute_uri(reverse('login'))
        }
        email_sent = send_business_credentials_email(email, email_context)
        
        if email_sent:
            messages.success(request, f'Business account created successfully for {branch_name}! Login credentials have been sent to {email}.')
        else:
            messages.warning(request, f'Business account created successfully for {branch_name}, but there was an issue sending the email with credentials.')
            
        return redirect('business_creation_success')
    
    context = {}
    return render(request, 'licenses/admin_create_business.html', context)


@login_required
@user_passes_test(lambda u: u.is_superuser)
def business_creation_success(request):
    """
    Success page after creating a business account
    """
    # Get data from session
    context = {
        'branch_name': request.session.get('branch_name', ''),
        'branch_address': request.session.get('branch_address', ''),
        'branch_email': request.session.get('branch_email', ''),
        'branch_phone': request.session.get('branch_phone', ''),
        'license_key': request.session.get('license_key', ''),
        'issue_date': request.session.get('issue_date', ''),
        'expiry_date': request.session.get('expiry_date', ''),
        'duration': request.session.get('duration', ''),
        'admin_name': request.session.get('admin_name', ''),
        'admin_email': request.session.get('admin_email', ''),
        'admin_password': request.session.get('admin_password', ''),
        'site_url': request.session.get('site_url', ''),
    }
    
    # Clear session data for security
    for key in ['branch_name', 'branch_address', 'branch_email', 'branch_phone', 
                'license_key', 'issue_date', 'expiry_date', 'duration', 
                'admin_name', 'admin_email', 'admin_password', 'site_url']:
        if key in request.session:
            del request.session[key]
    
    return render(request, 'licenses/business_creation_success.html', context)
