from django.test import TestCase
from django.contrib.auth import get_user_model
from django.urls import reverse
from .models import CompanySettings, Notification, AuditLog

User = get_user_model()

class CoreModelTests(TestCase):
    def setUp(self):
        self.user = User.objects.create_user(
            email='testuser@example.com',
            password='testpass123',
            first_name='Test',
            last_name='User',
            role='admin'
        )
    
    def test_company_settings(self):
        settings = CompanySettings.get_settings()
        self.assertEqual(settings.company_name, 'AsapLoads POS')
        
        # Test update
        settings.company_name = 'Updated Company'
        settings.save()
        
        updated_settings = CompanySettings.get_settings()
        self.assertEqual(updated_settings.company_name, 'Updated Company')
    
    def test_notification_creation(self):
        notification = Notification.objects.create(
            title='Test Notification',
            message='This is a test'
        )
        self.assertEqual(Notification.objects.count(), 1)
        self.assertEqual(notification.title, 'Test Notification')
        self.assertFalse(notification.is_read)
    
    def test_audit_log(self):
        log = AuditLog.log_action(
            user=self.user,
            action='create',
            entity_type='product',
            entity_id=1,
            details='Created a new product'
        )
        
        self.assertEqual(AuditLog.objects.count(), 1)
        self.assertEqual(log.user, self.user)
        self.assertEqual(log.action, 'create')
        self.assertEqual(log.entity_type, 'product')
        self.assertEqual(log.entity_id, 1)

class DashboardViewTest(TestCase):
    def setUp(self):
        self.user = User.objects.create_user(
            email='testuser@example.com',
            password='testpass123',
            first_name='Test',
            last_name='User',
            role='admin'
        )
        self.client.login(username='testuser@example.com', password='testpass123')
    
    def test_dashboard_access(self):
        response = self.client.get(reverse('dashboard'))
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'dashboard/index.html')
