from django.shortcuts import redirect
from django.urls import reverse
from django.contrib import messages

class TenantMiddleware:
    """
    Middleware for handling multi-tenancy and business isolation
    """
    def __init__(self, get_response):
        self.get_response = get_response

    def __call__(self, request):
        # Skip middleware for authentication paths and static resources
        if request.path.startswith('/accounts/login/') or \
           request.path.startswith('/accounts/logout/') or \
           request.path.startswith('/accounts/setup/') or \
           request.path.startswith('/admin/') or \
           request.path.startswith('/static/') or \
           request.path.startswith('/media/') or \
           request.path == '/favicon.ico':
            return self.get_response(request)
            
        # If user is not authenticated, just pass through
        if not request.user.is_authenticated:
            return self.get_response(request)
        
        # Set branch context for system owners and superusers
        if request.user.is_superuser or request.user.role == 'system_owner':
            # Check if there's a selected branch in the session
            branch_id = request.session.get('selected_branch_id')
            if branch_id:
                from branches.models import Branch
                try:
                    branch = Branch.objects.get(id=branch_id)
                    request.branch = branch
                except Branch.DoesNotExist:
                    if 'selected_branch_id' in request.session:
                        del request.session['selected_branch_id']
                    request.branch = getattr(request.user, 'branch', None)
            else:
                # Default to user's branch if available
                request.branch = getattr(request.user, 'branch', None)
        else:
            # For regular users, always use their assigned branch
            if not request.user.branch and not request.path.startswith('/accounts/'):
                messages.error(request, 'You are not assigned to any branch. Please contact your administrator.')
                return redirect('logout')
                
            # Check branch license for non-admin users
            if request.user.branch and request.user.role not in ['admin', 'system_owner'] and not request.path.startswith('/accounts/'):
                if not request.user.branch.has_valid_license():
                    messages.error(request, 'Your branch license has expired or is invalid. Please contact your administrator.')
                    return redirect('logout')
                    
            # Set branch context for the request
            request.branch = request.user.branch
        
        return self.get_response(request)
