from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required, user_passes_test
from django.contrib import messages
from django.http import JsonResponse
from django.db.models import Q
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger

from .models import Category, Product, StockMovement
from .forms import CategoryForm, ProductForm, ProductSearchForm, StockAdjustmentForm
from core.models import AuditLog

def is_admin_or_manager(user):
    """
    Check if user is an admin or manager
    """
    return user.is_authenticated and (user.role in ['admin', 'manager'])

@login_required
def product_list(request):
    """
    List all products, with filtering options
    """
    # Base queryset filtered by user's branch if not admin
    if request.user.role == 'admin':
        products = Product.objects.all().select_related('category', 'branch')
    else:
        products = Product.objects.filter(branch=request.user.branch).select_related('category', 'branch')
    
    # Initialize search form
    search_form = ProductSearchForm(request.user, request.GET)
    
    # Apply filters if form is valid
    if search_form.is_valid():
        query = search_form.cleaned_data.get('query')
        category = search_form.cleaned_data.get('category')
        
        if query:
            products = products.filter(
                Q(name__icontains=query) | 
                Q(sku__icontains=query) | 
                Q(barcode__icontains=query)
            )
        
        if category:
            products = products.filter(category=category)
    
    # Order products by name
    products = products.order_by('name')
    
    # Pagination
    paginator = Paginator(products, 10)  # Show 10 products per page
    page = request.GET.get('page')
    
    try:
        products = paginator.page(page)
    except PageNotAnInteger:
        products = paginator.page(1)
    except EmptyPage:
        products = paginator.page(paginator.num_pages)
    
    context = {
        'products': products,
        'search_form': search_form,
    }
    return render(request, 'products/product_list.html', context)

@login_required
def product_detail(request, product_id):
    """
    Show product details
    """
    # Get product, ensuring branch access
    if request.user.role == 'admin':
        product = get_object_or_404(Product, id=product_id)
    else:
        product = get_object_or_404(Product, id=product_id, branch=request.user.branch)
    
    # Get recent stock movements
    stock_movements = StockMovement.objects.filter(product=product).order_by('-timestamp')[:10]
    
    context = {
        'product': product,
        'stock_movements': stock_movements,
    }
    return render(request, 'products/product_detail.html', context)

@login_required
@user_passes_test(is_admin_or_manager)
def product_add(request):
    """
    Add a new product
    """
    if request.method == 'POST':
        form = ProductForm(request.POST, request.FILES)
        if form.is_valid():
            product = form.save(commit=False)
            # Handle the file upload
            if 'image' in request.FILES:
                product.image = request.FILES['image']
                
            # Ensure cost_price is defaulted to 0 if not provided
            if product.cost_price is None:
                product.cost_price = 0
                
            product.save()
            
            # Log the action
            AuditLog.log_action(
                user=request.user,
                action='create',
                entity_type='product',
                entity_id=product.id,
                details=f"Created product: {product.name}",
                ip_address=request.META.get('REMOTE_ADDR')
            )
            
            messages.success(request, f'Product "{product.name}" added successfully.')
            return redirect('product_list')
        else:
            # Debug information for form errors
            messages.error(request, f'Error adding product. Please check the form errors.')
            print(f"Form errors: {form.errors}")
    else:
        # Pre-select user's branch if not admin
        initial = {'cost_price': 0}  # Default cost_price to 0
        if request.user.role != 'admin' and request.user.branch:
            initial['branch'] = request.user.branch
        
        form = ProductForm(initial=initial)
        
        # Restrict branch selection if not admin
        if request.user.role != 'admin':
            form.fields['branch'].queryset = form.fields['branch'].queryset.filter(id=request.user.branch.id)
            form.fields['category'].queryset = form.fields['category'].queryset.filter(branch=request.user.branch)
    
    context = {
        'form': form,
        'title': 'Add Product',
    }
    return render(request, 'products/product_form.html', context)

@login_required
@user_passes_test(is_admin_or_manager)
def product_edit(request, product_id):
    """
    Edit an existing product
    """
    # Get product, ensuring branch access
    if request.user.role == 'admin':
        product = get_object_or_404(Product, id=product_id)
    else:
        product = get_object_or_404(Product, id=product_id, branch=request.user.branch)
    
    if request.method == 'POST':
        form = ProductForm(request.POST, request.FILES, instance=product)
        if form.is_valid():
            product = form.save(commit=False)
            # Handle the file upload
            if 'image' in request.FILES:
                product.image = request.FILES['image']
            
            # Ensure cost_price is defaulted to 0 if not provided
            if product.cost_price is None:
                product.cost_price = 0
                
            product.save()
            
            # Log the action
            AuditLog.log_action(
                user=request.user,
                action='update',
                entity_type='product',
                entity_id=product.id,
                details=f"Updated product: {product.name}",
                ip_address=request.META.get('REMOTE_ADDR')
            )
            
            messages.success(request, f'Product "{product.name}" updated successfully.')
            return redirect('product_list')
        else:
            # Debug information for form errors
            messages.error(request, f'Error updating product. Please check the form errors.')
            print(f"Form errors: {form.errors}")
    else:
        form = ProductForm(instance=product)
        
        # Restrict branch selection if not admin
        if request.user.role != 'admin':
            form.fields['branch'].queryset = form.fields['branch'].queryset.filter(id=request.user.branch.id)
            form.fields['category'].queryset = form.fields['category'].queryset.filter(branch=request.user.branch)
    
    context = {
        'form': form,
        'title': 'Edit Product',
        'product': product,
    }
    return render(request, 'products/product_form.html', context)

@login_required
@user_passes_test(is_admin_or_manager)
def product_delete(request, product_id):
    """
    Delete a product
    """
    # Get product, ensuring branch access
    if request.user.role == 'admin':
        product = get_object_or_404(Product, id=product_id)
    else:
        product = get_object_or_404(Product, id=product_id, branch=request.user.branch)
    
    if request.method == 'POST':
        product_name = product.name
        product.delete()
        
        # Log the action
        AuditLog.log_action(
            user=request.user,
            action='delete',
            entity_type='product',
            entity_id=product_id,
            details=f"Deleted product: {product_name}",
            ip_address=request.META.get('REMOTE_ADDR')
        )
        
        messages.success(request, f'Product "{product_name}" has been deleted.')
        return redirect('product_list')
    
    context = {
        'product': product,
    }
    return render(request, 'products/product_confirm_delete.html', context)

@login_required
@user_passes_test(is_admin_or_manager)
def adjust_stock(request, product_id):
    """
    Adjust product stock
    """
    # Get product, ensuring branch access
    if request.user.role == 'admin':
        product = get_object_or_404(Product, id=product_id)
    else:
        product = get_object_or_404(Product, id=product_id, branch=request.user.branch)
    
    if request.method == 'POST':
        form = StockAdjustmentForm(request.POST)
        if form.is_valid():
            adjustment_type = form.cleaned_data['adjustment_type']
            quantity = form.cleaned_data['quantity']
            reason = form.cleaned_data['reason']
            
            product.adjust_stock(adjustment_type, quantity, request.user, reason)
            
            # Log the action
            AuditLog.log_action(
                user=request.user,
                action='update',
                entity_type='product_stock',
                entity_id=product.id,
                details=f"{adjustment_type.capitalize()} {quantity} to stock of {product.name}. Reason: {reason}",
                ip_address=request.META.get('REMOTE_ADDR')
            )
            
            messages.success(request, f'Stock for "{product.name}" adjusted successfully.')
            return redirect('product_detail', product_id=product.id)
    else:
        form = StockAdjustmentForm()
    
    context = {
        'form': form,
        'product': product,
        'title': 'Adjust Stock',
    }
    return render(request, 'products/stock_adjustment_form.html', context)

@login_required
def product_search_api(request):
    """
    API for searching products, used in sale creation
    Ajax API that accepts requests with proper CSRF token
    """
    query = request.GET.get('q', '')
    all_products = request.GET.get('all', 'false').lower() == 'true'
    
    # Base queryset filtered by user's branch if not admin
    if request.user.role == 'admin':
        products = Product.objects.filter(is_active=True)
    else:
        products = Product.objects.filter(is_active=True, branch=request.user.branch)
    
    # Apply search filter if not requesting all products
    if query and not all_products:
        products = products.filter(
            Q(name__icontains=query) | 
            Q(sku__icontains=query) | 
            Q(barcode__icontains=query)
        )
    
    # Order products by name
    products = products.order_by('name')
    
    # Create list with all necessary data including image URLs
    products_list = []
    for product in products:
        product_data = {
            'id': product.id,
            'name': product.name,
            'price': float(product.price),
            'quantity': product.quantity,
            'sku': product.sku,
            'barcode': product.barcode,
            'image_url': product.image.url if product.image else None
        }
        products_list.append(product_data)
    
    return JsonResponse({'products': products_list})

@login_required
def category_list(request):
    """
    List all categories
    """
    # Base queryset filtered by user's branch if not admin
    if request.user.role == 'admin':
        categories = Category.objects.all().select_related('branch')
    else:
        categories = Category.objects.filter(branch=request.user.branch).select_related('branch')
    
    context = {
        'categories': categories,
    }
    return render(request, 'products/category_list.html', context)

@login_required
@user_passes_test(is_admin_or_manager)
def category_add(request):
    """
    Add a new category
    """
    if request.method == 'POST':
        form = CategoryForm(request.POST)
        if form.is_valid():
            category = form.save()
            
            # Log the action
            AuditLog.log_action(
                user=request.user,
                action='create',
                entity_type='category',
                entity_id=category.id,
                details=f"Created category: {category.name}",
                ip_address=request.META.get('REMOTE_ADDR')
            )
            
            messages.success(request, f'Category "{category.name}" added successfully.')
            return redirect('category_list')
    else:
        # Pre-select user's branch if not admin
        initial = {}
        if request.user.role != 'admin' and request.user.branch:
            initial['branch'] = request.user.branch
        
        form = CategoryForm(initial=initial)
        
        # Restrict branch selection if not admin
        if request.user.role != 'admin':
            form.fields['branch'].queryset = form.fields['branch'].queryset.filter(id=request.user.branch.id)
    
    context = {
        'form': form,
        'title': 'Add Category',
    }
    return render(request, 'products/category_form.html', context)

@login_required
@user_passes_test(is_admin_or_manager)
def category_edit(request, category_id):
    """
    Edit an existing category
    """
    # Get category, ensuring branch access
    if request.user.role == 'admin':
        category = get_object_or_404(Category, id=category_id)
    else:
        category = get_object_or_404(Category, id=category_id, branch=request.user.branch)
    
    if request.method == 'POST':
        form = CategoryForm(request.POST, instance=category)
        if form.is_valid():
            category = form.save()
            
            # Log the action
            AuditLog.log_action(
                user=request.user,
                action='update',
                entity_type='category',
                entity_id=category.id,
                details=f"Updated category: {category.name}",
                ip_address=request.META.get('REMOTE_ADDR')
            )
            
            messages.success(request, f'Category "{category.name}" updated successfully.')
            return redirect('category_list')
    else:
        form = CategoryForm(instance=category)
        
        # Restrict branch selection if not admin
        if request.user.role != 'admin':
            form.fields['branch'].queryset = form.fields['branch'].queryset.filter(id=request.user.branch.id)
    
    context = {
        'form': form,
        'title': 'Edit Category',
        'category': category,
    }
    return render(request, 'products/category_form.html', context)

@login_required
@user_passes_test(is_admin_or_manager)
def category_delete(request, category_id):
    """
    Delete a category
    """
    # Get category, ensuring branch access
    if request.user.role == 'admin':
        category = get_object_or_404(Category, id=category_id)
    else:
        category = get_object_or_404(Category, id=category_id, branch=request.user.branch)
    
    # Check if category has products
    if category.product_set.exists():
        messages.error(request, f'Cannot delete category "{category.name}" because it has products assigned to it.')
        return redirect('category_list')
    
    if request.method == 'POST':
        category_name = category.name
        category.delete()
        
        # Log the action
        AuditLog.log_action(
            user=request.user,
            action='delete',
            entity_type='category',
            entity_id=category_id,
            details=f"Deleted category: {category_name}",
            ip_address=request.META.get('REMOTE_ADDR')
        )
        
        messages.success(request, f'Category "{category_name}" has been deleted.')
        return redirect('category_list')
    
    context = {
        'category': category,
    }
    return render(request, 'products/category_confirm_delete.html', context)
