from django.contrib import admin
from django.utils.html import format_html
from django.urls import reverse
from django.utils.safestring import mark_safe

from .models import (
    SupplierCategory, Supplier, SupplierContact, 
    SupplierNote, SupplierDocument
)

class SupplierContactInline(admin.TabularInline):
    model = SupplierContact
    extra = 1
    fields = ('first_name', 'last_name', 'position', 'email', 'phone', 'is_primary')
    
    def has_delete_permission(self, request, obj=None):
        return True

class SupplierDocumentInline(admin.TabularInline):
    model = SupplierDocument
    extra = 1
    fields = ('document_type', 'title', 'file', 'valid_from', 'valid_to')
    readonly_fields = ('uploaded_at',)

class SupplierNoteInline(admin.TabularInline):
    model = SupplierNote
    extra = 1
    fields = ('note', 'created_by', 'created_at')
    readonly_fields = ('created_by', 'created_at')
    
    def has_delete_permission(self, request, obj=None):
        return request.user.is_superuser

@admin.register(SupplierCategory)
class SupplierCategoryAdmin(admin.ModelAdmin):
    list_display = ('name', 'supplier_count', 'is_active', 'created_at')
    list_filter = ('is_active',)
    search_fields = ('name', 'description')
    prepopulated_fields = {}
    
    def supplier_count(self, obj):
        return obj.suppliers.count()
    supplier_count.short_description = 'Suppliers'

@admin.register(Supplier)
class SupplierAdmin(admin.ModelAdmin):
    list_display = ('name', 'supplier_type', 'email', 'phone', 'city', 'country', 'is_active', 'preferred')
    list_filter = ('supplier_type', 'is_active', 'preferred', 'category', 'country')
    search_fields = ('name', 'company_name', 'email', 'phone', 'tax_identification_number', 'vat_number')
    list_editable = ('is_active', 'preferred')
    readonly_fields = ('created_at', 'updated_at', 'account_balance', 'total_purchases', 'total_payments')
    inlines = [SupplierContactInline, SupplierDocumentInline, SupplierNoteInline]
    
    fieldsets = (
        ('Basic Information', {
            'fields': (
                'name', 'supplier_type', 'company_name', 
                'is_active', 'preferred', 'category', 'branches'
            )
        }),
        ('Contact Information', {
            'fields': (
                'email', 'phone', 'mobile', 'website',
                'address_line1', 'address_line2',
                'city', 'state', 'postal_code', 'country'
            ),
            'classes': ('collapse',)
        }),
        ('Business Information', {
            'fields': (
                'tax_identification_number', 'registration_number', 
                'vat_number', 'currency', 'payment_terms', 'credit_limit'
            ),
            'classes': ('collapse',)
        }),
        ('Financial Information', {
            'fields': (
                'account_balance', 'total_purchases', 'total_payments'
            ),
            'classes': ('collapse',)
        }),
        ('Audit Information', {
            'fields': ('notes', 'created_by', 'created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )
    
    def save_model(self, request, obj, form, change):
        if not obj.pk:
            obj.created_by = request.user
        super().save_model(request, obj, form, change)
    
    def get_queryset(self, request):
        return super().get_queryset(request).prefetch_related('branches', 'category')

@admin.register(SupplierContact)
class SupplierContactAdmin(admin.ModelAdmin):
    list_display = ('name', 'supplier_link', 'position', 'email', 'phone', 'is_primary')
    list_filter = ('is_primary', 'contact_type')
    search_fields = ('first_name', 'last_name', 'email', 'phone', 'supplier__name')
    list_select_related = ('supplier',)
    
    def supplier_link(self, obj):
        url = reverse('admin:suppliers_supplier_change', args=[obj.supplier.id])
        return mark_safe(f'<a href="{url}">{obj.supplier.name}</a>')
    supplier_link.short_description = 'Supplier'
    supplier_link.admin_order_field = 'supplier__name'
    
    def name(self, obj):
        return f"{obj.first_name} {obj.last_name}"
    name.admin_order_field = 'last_name'

@admin.register(SupplierDocument)
class SupplierDocumentAdmin(admin.ModelAdmin):
    list_display = ('title', 'supplier_link', 'document_type', 'file_link', 'is_valid', 'valid_from', 'valid_to')
    list_filter = ('document_type', 'uploaded_at', 'valid_to')
    search_fields = ('title', 'description', 'supplier__name')
    readonly_fields = ('uploaded_at', 'uploaded_by')
    list_select_related = ('supplier', 'uploaded_by')
    
    def supplier_link(self, obj):
        url = reverse('admin:suppliers_supplier_change', args=[obj.supplier.id])
        return mark_safe(f'<a href="{url}">{obj.supplier.name}</a>')
    supplier_link.short_description = 'Supplier'
    supplier_link.admin_order_field = 'supplier__name'
    
    def file_link(self, obj):
        if obj.file:
            return mark_safe(f'<a href="{obj.file.url}" target="_blank">View Document</a>')
        return "-"
    file_link.short_description = 'File'
    
    def save_model(self, request, obj, form, change):
        if not obj.pk:
            obj.uploaded_by = request.user
        super().save_model(request, obj, form, change)

@admin.register(SupplierNote)
class SupplierNoteAdmin(admin.ModelAdmin):
    list_display = ('truncated_note', 'supplier_link', 'created_by', 'created_at')
    search_fields = ('note', 'supplier__name', 'created_by__username')
    list_filter = ('created_at',)
    readonly_fields = ('created_by', 'created_at', 'updated_at')
    list_select_related = ('supplier', 'created_by')
    
    def supplier_link(self, obj):
        url = reverse('admin:suppliers_supplier_change', args=[obj.supplier.id])
        return mark_safe(f'<a href="{url}">{obj.supplier.name}</a>')
    supplier_link.short_description = 'Supplier'
    supplier_link.admin_order_field = 'supplier__name'
    
    def truncated_note(self, obj):
        return obj.note[:100] + '...' if len(obj.note) > 100 else obj.note
    truncated_note.short_description = 'Note'
