from django.test import TestCase
from django.urls import reverse
from django.contrib.auth import get_user_model
from decimal import Decimal
from .models import Sale, SaleItem, Payment
from products.models import Product, Category
from customers.models import Customer
from branches.models import Branch

User = get_user_model()

class SaleModelTest(TestCase):
    def setUp(self):
        self.branch = Branch.objects.create(
            name='Test Branch',
            address='123 Test St'
        )
        
        self.category = Category.objects.create(
            name='Test Category',
            branch=self.branch
        )
        
        self.product = Product.objects.create(
            name='Test Product',
            category=self.category,
            branch=self.branch,
            price=Decimal('10.00'),
            quantity=50
        )
        
        self.customer = Customer.objects.create(
            first_name='John',
            last_name='Doe',
            branch=self.branch
        )
        
        self.cashier = User.objects.create_user(
            email='cashier@example.com',
            password='testpassword',
            first_name='Cash',
            last_name='Ier',
            role='cashier',
            branch=self.branch
        )
        
        self.sale = Sale.objects.create(
            customer=self.customer,
            cashier=self.cashier,
            branch=self.branch
        )
    
    def test_invoice_number_generation(self):
        self.assertIsNotNone(self.sale.invoice_number)
        self.assertTrue(self.sale.invoice_number.startswith('INV-'))
    
    def test_sale_items_and_totals(self):
        # Add item to sale
        item = SaleItem.objects.create(
            sale=self.sale,
            product=self.product,
            quantity=5,
            price=Decimal('10.00'),
            update_stock=False  # Don't update stock in test
        )
        
        # Refresh sale from database
        self.sale.refresh_from_db()
        
        # Check totals
        self.assertEqual(self.sale.subtotal, Decimal('50.00'))
        self.assertEqual(self.sale.total_amount, Decimal('50.00'))
        
        # Check item total
        self.assertEqual(item.total, Decimal('50.00'))
    
    def test_payment_and_status_updates(self):
        # Add item to sale
        SaleItem.objects.create(
            sale=self.sale,
            product=self.product,
            quantity=5,
            price=Decimal('10.00'),
            update_stock=False  # Don't update stock in test
        )
        
        # Refresh sale from database
        self.sale.refresh_from_db()
        
        # Initial status should be pending
        self.assertEqual(self.sale.payment_status, 'pending')
        
        # Add partial payment
        Payment.objects.create(
            sale=self.sale,
            amount=Decimal('25.00'),
            payment_method='cash'
        )
        
        # Refresh sale from database
        self.sale.refresh_from_db()
        
        # Status should be partial
        self.assertEqual(self.sale.payment_status, 'partial')
        self.assertEqual(self.sale.total_paid, Decimal('25.00'))
        self.assertEqual(self.sale.balance_due, Decimal('25.00'))
        
        # Add remaining payment
        Payment.objects.create(
            sale=self.sale,
            amount=Decimal('25.00'),
            payment_method='card'
        )
        
        # Refresh sale from database
        self.sale.refresh_from_db()
        
        # Status should be paid
        self.assertEqual(self.sale.payment_status, 'paid')
        self.assertEqual(self.sale.total_paid, Decimal('50.00'))
        self.assertEqual(self.sale.balance_due, Decimal('0.00'))

class SaleViewTest(TestCase):
    def setUp(self):
        self.branch = Branch.objects.create(
            name='Test Branch',
            address='123 Test St'
        )
        
        self.cashier = User.objects.create_user(
            email='cashier@example.com',
            password='testpassword',
            first_name='Cash',
            last_name='Ier',
            role='cashier',
            branch=self.branch
        )
        
        self.client.login(username='cashier@example.com', password='testpassword')
        
        # Create a product for testing
        self.category = Category.objects.create(
            name='Test Category',
            branch=self.branch
        )
        
        self.product = Product.objects.create(
            name='Test Product',
            category=self.category,
            branch=self.branch,
            price=Decimal('10.00'),
            quantity=50
        )
    
    def test_new_sale_view(self):
        response = self.client.get(reverse('new_sale'))
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'sales/new_sale.html')
