from django.db import models
from django.urls import reverse
from django.utils.text import slugify
from django.conf import settings
from django.db.models import Q, UniqueConstraint

class Category(models.Model):
    """
    Product category model
    """
    name = models.CharField(max_length=100)
    description = models.TextField(blank=True)
    branch = models.ForeignKey('branches.Branch', on_delete=models.CASCADE, null=True, blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name_plural = 'Categories'
        ordering = ['name']
        unique_together = ['name', 'branch']  # Keep this if still needed

    def __str__(self):
        return self.name

    def get_absolute_url(self):
        return reverse('category_detail', args=[self.pk])


class Product(models.Model):
    """
    Product model
    """
    name = models.CharField(max_length=100)
    description = models.TextField(blank=True)
    category = models.ForeignKey(Category, on_delete=models.SET_NULL, null=True, blank=True)
    branch = models.ForeignKey('branches.Branch', on_delete=models.CASCADE)
    price = models.DecimalField(max_digits=10, decimal_places=2)
    cost_price = models.DecimalField(max_digits=10, decimal_places=2, default=0)
    quantity = models.IntegerField(default=0)
    low_stock_threshold = models.IntegerField(default=10)
    sku = models.CharField(max_length=50, blank=True, null=True)  # ✅ allow NULL
    barcode = models.CharField(max_length=50, blank=True, null=True)  # ✅ allow NULL
    image = models.ImageField(upload_to='products/', null=True, blank=True)
    is_active = models.BooleanField(default=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ['name']
        constraints = [
            # ✅ Uniqueness only enforced when value is not NULL
            UniqueConstraint(fields=['sku', 'branch'], name='unique_sku_per_branch', condition=Q(sku__isnull=False)),
            UniqueConstraint(fields=['barcode', 'branch'], name='unique_barcode_per_branch', condition=Q(barcode__isnull=False)),
        ]

    def __str__(self):
        return self.name

    def get_absolute_url(self):
        return reverse('product_detail', args=[self.pk])

    @property
    def is_low_stock(self):
        """
        Check if product is low on stock
        """
        return self.quantity <= self.low_stock_threshold

    @property
    def profit_margin(self):
        """
        Calculate profit margin percentage
        """
        if self.cost_price == 0:
            return 0
        profit = self.price - self.cost_price
        return (profit / self.cost_price) * 100

    def adjust_stock(self, adjustment_type, quantity, user=None, reason=''):
        """
        Adjust product stock and create a stock movement record
        """
        original_quantity = self.quantity

        if adjustment_type == 'add':
            self.quantity += quantity
        elif adjustment_type == 'remove':
            self.quantity = max(0, self.quantity - quantity)
        elif adjustment_type == 'set':
            self.quantity = quantity

        self.save()

        # Create stock movement record
        StockMovement.objects.create(
            product=self,
            user=user,
            from_quantity=original_quantity,
            to_quantity=self.quantity,
            adjustment_quantity=quantity,
            adjustment_type=adjustment_type,
            reason=reason
        )

        # Check if stock is low after adjustment
        if self.is_low_stock:
            from core.models import Notification
            Notification.create_low_stock_notification(self, self.branch)

        return self.quantity


class StockMovement(models.Model):
    """
    Record of product stock adjustments
    """
    ADJUSTMENT_TYPES = [
        ('add', 'Add to Stock'),
        ('remove', 'Remove from Stock'),
        ('set', 'Set Stock Level'),
        ('sale', 'Sale'),
        ('purchase', 'Purchase'),
        ('return', 'Return'),
    ]
    
    product = models.ForeignKey(Product, on_delete=models.CASCADE)
    user = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.SET_NULL, null=True, blank=True)
    from_quantity = models.IntegerField()
    to_quantity = models.IntegerField()
    adjustment_quantity = models.IntegerField()
    adjustment_type = models.CharField(max_length=10, choices=ADJUSTMENT_TYPES)
    reason = models.TextField(blank=True)
    timestamp = models.DateTimeField(auto_now_add=True)

    class Meta:
        ordering = ['-timestamp']

    def __str__(self):
        return f"{self.adjustment_type} {self.adjustment_quantity} of {self.product.name}"
