from django.db import models
from django.urls import reverse

class Customer(models.Model):
    """
    Customer model for storing customer information
    """
    first_name = models.CharField(max_length=100)
    last_name = models.CharField(max_length=100)
    email = models.EmailField(blank=True)
    phone = models.CharField(max_length=20, blank=True)
    address = models.CharField(max_length=200, blank=True)
    city = models.CharField(max_length=100, blank=True)
    state = models.CharField(max_length=100, blank=True)
    postal_code = models.CharField(max_length=20, blank=True)
    branch = models.ForeignKey('branches.Branch', on_delete=models.CASCADE)
    notes = models.TextField(blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    class Meta:
        ordering = ['first_name', 'last_name']
    
    def __str__(self):
        return f"{self.first_name} {self.last_name}"
    
    def get_absolute_url(self):
        return reverse('customer_detail', args=[self.pk])
    
    @property
    def full_name(self):
        """
        Returns the customer's full name
        """
        return f"{self.first_name} {self.last_name}"
    
    @property
    def full_address(self):
        """
        Returns the customer's full address
        """
        address_parts = [self.address]
        
        if self.city:
            address_parts.append(self.city)
        
        if self.state and self.postal_code:
            address_parts.append(f"{self.state}, {self.postal_code}")
        elif self.state:
            address_parts.append(self.state)
        elif self.postal_code:
            address_parts.append(self.postal_code)
        
        return ", ".join(filter(None, address_parts))
    
    @property
    def total_purchases(self):
        """
        Returns the total number of purchases made by the customer
        """
        return self.sale_set.count()
    
    @property
    def total_spent(self):
        """
        Returns the total amount spent by the customer
        """
        from django.db.models import Sum
        result = self.sale_set.filter(payment_status='paid').aggregate(total=Sum('total_amount'))
        return result['total'] or 0
