from django.test import TestCase
from django.utils import timezone
from datetime import timedelta
from .models import License
from branches.models import Branch

class LicenseModelTest(TestCase):
    def setUp(self):
        self.branch = Branch.objects.create(
            name='Test Branch',
            address='123 Test St'
        )
    
    def test_license_generation(self):
        license_data = License.generate_license_key(self.branch.id)
        
        self.assertIn('license_key', license_data)
        self.assertIn('issue_date', license_data)
        self.assertIn('expiry_date', license_data)
        
        # The license key should have the ASAP prefix
        self.assertTrue(license_data['license_key'].startswith('ASAP-'))
        
        # The expiry date should be in the future
        self.assertGreater(license_data['expiry_date'], timezone.now().date())
    
    def test_license_creation(self):
        license_data = License.generate_license_key(self.branch.id, days=30)
        
        license = License.objects.create(
            license_key=license_data['license_key'],
            branch=self.branch,
            issue_date=license_data['issue_date'],
            expiry_date=license_data['expiry_date']
        )
        
        self.assertEqual(License.objects.count(), 1)
        self.assertEqual(license.branch, self.branch)
        self.assertTrue(license.is_active)
        self.assertEqual(license.days_remaining, 30)
    
    def test_license_expiry(self):
        # Create an expired license
        yesterday = timezone.now().date() - timedelta(days=1)
        
        license = License.objects.create(
            license_key='EXPIRED-LICENSE-KEY',
            branch=self.branch,
            issue_date=yesterday - timedelta(days=30),
            expiry_date=yesterday
        )
        
        self.assertTrue(license.is_expired)
        self.assertEqual(license.days_remaining, 0)
    
    def test_license_revocation(self):
        license_data = License.generate_license_key(self.branch.id)
        
        license = License.objects.create(
            license_key=license_data['license_key'],
            branch=self.branch,
            issue_date=license_data['issue_date'],
            expiry_date=license_data['expiry_date']
        )
        
        self.assertTrue(license.is_active)
        
        # Revoke the license
        revoked = License.revoke_license(license.id)
        license.refresh_from_db()
        
        self.assertTrue(revoked)
        self.assertFalse(license.is_active)
