from django.db import models
from django.core.validators import MinValueValidator
from django.contrib.auth import get_user_model
from django.utils.translation import gettext_lazy as _
from django.utils import timezone
from decimal import Decimal

User = get_user_model()

class ExpenseCategory(models.Model):
    name = models.CharField(_('Category Name'), max_length=100, unique=True)
    description = models.TextField(_('Description'), blank=True)
    is_active = models.BooleanField(_('Active'), default=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name = _('Expense Category')
        verbose_name_plural = _('Expense Categories')
        ordering = ['name']

    def __str__(self):
        return self.name

class Expense(models.Model):
    STATUS_CHOICES = [
        ('draft', _('Draft')),
        ('pending', _('Pending Approval')),
        ('approved', _('Approved')),
        ('rejected', _('Rejected')),
        ('paid', _('Paid')),
    ]

    PAYMENT_METHODS = [
        ('cash', _('Cash')),
        ('bank_transfer', _('Bank Transfer')),
        ('credit_card', _('Credit Card')),
        ('mobile_money', _('Mobile Money')),
        ('other', _('Other')),
    ]

    reference = models.CharField(_('Reference'), max_length=50, unique=True)
    description = models.TextField(_('Description'))
    category = models.ForeignKey(
        ExpenseCategory,
        on_delete=models.PROTECT,
        verbose_name=_('Category'),
        related_name='expenses'
    )
    amount = models.DecimalField(
        _('Amount'),
        max_digits=12,
        decimal_places=2,
        validators=[MinValueValidator(Decimal('0.01'))]
    )
    expense_date = models.DateField(_('Expense Date'), default=timezone.now)
    payment_method = models.CharField(
        _('Payment Method'),
        max_length=20,
        choices=PAYMENT_METHODS,
        default='cash'
    )
    status = models.CharField(
        _('Status'),
        max_length=20,
        choices=STATUS_CHOICES,
        default='draft'
    )
    created_by = models.ForeignKey(
        User,
        on_delete=models.PROTECT,
        related_name='expenses_created',
        verbose_name=_('Created By')
    )
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    notes = models.TextField(_('Notes'), blank=True)

    class Meta:
        verbose_name = _('Expense')
        verbose_name_plural = _('Expenses')
        ordering = ['-expense_date', '-created_at']
        permissions = [
            ('approve_expense', 'Can approve expense'),
            ('view_all_expenses', 'Can view all expenses'),
        ]

    def __str__(self):
        return f"{self.reference} - {self.category.name} - {self.amount}"

    def save(self, *args, **kwargs):
        if not self.reference:
            # Generate a reference if not provided
            last_expense = Expense.objects.order_by('-id').first()
            last_id = last_expense.id if last_expense else 0
            self.reference = f"EXP-{last_id + 1:05d}"
        super().save(*args, **kwargs)
