from django.shortcuts import render, redirect
from django.contrib.auth.decorators import login_required, user_passes_test
from django.db.models import Sum, Count, F, DecimalField
from django.db.models.functions import TruncDate, TruncMonth
from django.contrib import messages
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger
from django.utils import timezone
from datetime import timedelta
from expenses.models import Expense
from purchases.models import PurchaseOrder, PurchaseOrderItem, GoodsReceipt
from quotations.models import Quotation
from suppliers.models import Supplier
from django.db.models import Q

# Utility function to check if user is admin or manager
def is_admin_or_manager(user):
    return user.is_authenticated and (user.role in ['admin', 'manager'] or user.is_superuser)

@login_required
@user_passes_test(is_admin_or_manager)
def expenses_report(request):
    """
    Expense report view for admin and managers
    """
    today = timezone.now().date()
    start_date = request.GET.get('start_date', (today - timedelta(days=30)).isoformat())
    end_date = request.GET.get('end_date', today.isoformat())
    
    try:
        expenses = Expense.objects.filter(
            date__range=[start_date, end_date]
        ).select_related('category', 'user').order_by('-date')
        
        # Filter by category if specified
        category = request.GET.get('category')
        if category:
            expenses = expenses.filter(category_id=category)
            
        # Calculate totals
        total_amount = expenses.aggregate(total=Sum('amount'))['total'] or 0
        
        # Get top categories
        categories = expenses.values('category__name').annotate(
            total=Sum('amount'),
            count=Count('id')
        ).order_by('-total')
        
        # Pagination
        page = request.GET.get('page', 1)
        paginator = Paginator(expenses, 25)
        
        try:
            expenses = paginator.page(page)
        except PageNotAnInteger:
            expenses = paginator.page(1)
        except EmptyPage:
            expenses = paginator.page(paginator.num_pages)
            
        context = {
            'expenses': expenses,
            'total_amount': total_amount,
            'categories': categories,
            'start_date': start_date,
            'end_date': end_date,
        }
        return render(request, 'reports/expenses_report.html', context)
        
    except Exception as e:
        messages.error(request, f"Error generating expense report: {str(e)}")
        return redirect('reports:dashboard')

@login_required
@user_passes_test(is_admin_or_manager)
def purchases_report(request):
    """
    Purchase report view for admin and managers
    """
    today = timezone.now().date()
    start_date = request.GET.get('start_date', (today - timedelta(days=30)).isoformat())
    end_date = request.GET.get('end_date', today.isoformat())
    
    try:
        # Convert string dates to date objects for comparison
        start_date_obj = timezone.datetime.strptime(start_date, '%Y-%m-%d').date()
        end_date_obj = timezone.datetime.strptime(end_date, '%Y-%m-%d').date()
        
        # Get purchase orders in the date range
        purchase_orders = PurchaseOrder.objects.filter(
            order_date__range=[start_date_obj, end_date_obj]
        ).select_related('supplier', 'branch').order_by('-order_date')
        
        # Filter by supplier if specified
        supplier_id = request.GET.get('supplier')
        if supplier_id:
            purchase_orders = purchase_orders.filter(supplier_id=supplier_id)
            
        # Calculate totals using the total_amount method
        total_amount = 0
        for po in purchase_orders:
            total_amount += po.total_amount
            
        # Get top suppliers
        suppliers = Supplier.objects.annotate(
            total_spent=Sum('purchase_orders__items__total_price'),
            order_count=Count('purchase_orders', distinct=True)
        ).filter(order_count__gt=0).order_by('-total_spent')
        
        # Pagination
        page = request.GET.get('page', 1)
        paginator = Paginator(purchase_orders, 25)
        
        try:
            purchase_orders = paginator.page(page)
        except PageNotAnInteger:
            purchase_orders = paginator.page(1)
        except EmptyPage:
            purchase_orders = paginator.page(paginator.num_pages)
            
        context = {
            'purchases': purchase_orders,  # Keep using 'purchases' in template for consistency
            'total_amount': total_amount,
            'suppliers': suppliers,
            'start_date': start_date,
            'end_date': end_date,
        }
        return render(request, 'reports/purchases_report.html', context)
        
    except Exception as e:
        messages.error(request, f"Error generating purchases report: {str(e)}")
        return redirect('reports:dashboard')

@login_required
@user_passes_test(is_admin_or_manager)
def quotations_report(request):
    """
    Quotations report view for admin and managers
    """
    today = timezone.now().date()
    start_date = request.GET.get('start_date', (today - timedelta(days=30)).isoformat())
    end_date = request.GET.get('end_date', today.isoformat())
    
    try:
        quotations = Quotation.objects.filter(
            created_at__date__range=[start_date, end_date]
        ).select_related('customer', 'created_by').order_by('-created_at')
        
        # Filter by status if specified
        status = request.GET.get('status')
        if status:
            quotations = quotations.filter(status=status)
            
        # Calculate totals and counts
        total_amount = quotations.aggregate(total=Sum('total_amount'))['total'] or 0
        status_counts = quotations.values('status').annotate(count=Count('id'))
        
        # Pagination
        page = request.GET.get('page', 1)
        paginator = Paginator(quotations, 25)
        
        try:
            quotations = paginator.page(page)
        except PageNotAnInteger:
            quotations = paginator.page(1)
        except EmptyPage:
            quotations = paginator.page(paginator.num_pages)
            
        context = {
            'quotations': quotations,
            'total_amount': total_amount,
            'status_counts': status_counts,
            'start_date': start_date,
            'end_date': end_date,
        }
        return render(request, 'reports/quotations_report.html', context)
        
    except Exception as e:
        messages.error(request, f"Error generating quotations report: {str(e)}")
        return redirect('reports:dashboard')

@login_required
@user_passes_test(is_admin_or_manager)
def suppliers_report(request):
    """
    Suppliers report view for admin and managers
    """
    try:
        suppliers = Supplier.objects.annotate(
            purchase_count=Count('purchases'),
            total_purchased=Sum('purchases__total_amount'),
            last_purchase_date=Max('purchases__date')
        ).order_by('-total_purchased')
        
        # Filter by search query if provided
        search = request.GET.get('search')
        if search:
            suppliers = suppliers.filter(
                Q(name__icontains=search) |
                Q(contact_person__icontains=search) |
                Q(email__icontains=search) |
                Q(phone__icontains=search)
            )
            
        # Calculate summary stats
        total_suppliers = suppliers.count()
        active_suppliers = suppliers.filter(is_active=True).count()
        
        # Pagination
        page = request.GET.get('page', 1)
        paginator = Paginator(suppliers, 25)
        
        try:
            suppliers = paginator.page(page)
        except PageNotAnInteger:
            suppliers = paginator.page(1)
        except EmptyPage:
            suppliers = paginator.page(paginator.num_pages)
            
        context = {
            'suppliers': suppliers,
            'total_suppliers': total_suppliers,
            'active_suppliers': active_suppliers,
            'search_query': search or '',
        }
        return render(request, 'reports/suppliers_report.html', context)
        
    except Exception as e:
        messages.error(request, f"Error generating suppliers report: {str(e)}")
        return redirect('reports:dashboard')
