from django.db import models
from django.conf import settings
from django.utils import timezone
from suppliers.models import Supplier
from products.models import Product
from branches.models import Branch

# Get the user model from settings
User = settings.AUTH_USER_MODEL

class PurchaseOrder(models.Model):
    PENDING = 'pending'
    COMPLETED = 'completed'
    CANCELLED = 'cancelled'
    
    STATUS_CHOICES = [
        (PENDING, 'Pending'),
        (COMPLETED, 'Completed'),
        (CANCELLED, 'Cancelled'),
    ]
    
    po_number = models.CharField(max_length=50, unique=True)
    supplier = models.ForeignKey(Supplier, on_delete=models.PROTECT, related_name='purchase_orders')
    order_date = models.DateField(default=timezone.now)
    expected_delivery_date = models.DateField(null=True, blank=True)
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default=PENDING)
    notes = models.TextField(blank=True)
    branch = models.ForeignKey(Branch, on_delete=models.PROTECT, null=True, blank=True)
    created_by = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.PROTECT, related_name='created_purchases')
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    class Meta:
        ordering = ['-order_date', '-created_at']
        
    def __str__(self):
        return f"PO-{self.po_number}"
        
    def save(self, *args, **kwargs):
        if not self.po_number:
            # Generate a new PO number if one doesn't exist
            last_po = PurchaseOrder.objects.order_by('-id').first()
            if last_po and last_po.po_number.isdigit():
                self.po_number = str(int(last_po.po_number) + 1).zfill(6)
            else:
                self.po_number = '100001'  # Starting PO number
        return super().save(*args, **kwargs)
    
    @property
    def total_amount(self):
        return sum(item.total_price for item in self.items.all())

class PurchaseOrderItem(models.Model):
    purchase_order = models.ForeignKey(PurchaseOrder, on_delete=models.CASCADE, related_name='items')
    product = models.ForeignKey(Product, on_delete=models.PROTECT)
    quantity = models.DecimalField(max_digits=10, decimal_places=2)
    unit_price = models.DecimalField(max_digits=10, decimal_places=2)
    total_price = models.DecimalField(max_digits=12, decimal_places=2, editable=False)
    received_quantity = models.DecimalField(max_digits=10, decimal_places=2, default=0)
    
    def save(self, *args, **kwargs):
        self.total_price = self.quantity * self.unit_price
        super().save(*args, **kwargs)
        
    def __str__(self):
        return f"{self.quantity} x {self.product.name} - ${self.unit_price} each"

class GoodsReceipt(models.Model):
    grn_number = models.CharField(max_length=50, unique=True)
    purchase_order = models.ForeignKey(PurchaseOrder, on_delete=models.PROTECT, related_name='goods_receipts')
    received_date = models.DateField(default=timezone.now)
    received_by = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.PROTECT)
    notes = models.TextField(blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        ordering = ['-received_date', '-created_at']
        
    def __str__(self):
        return f"GRN-{self.grn_number}"

class GoodsReceiptItem(models.Model):
    goods_receipt = models.ForeignKey(GoodsReceipt, on_delete=models.CASCADE, related_name='items')
    purchase_order_item = models.ForeignKey(PurchaseOrderItem, on_delete=models.PROTECT)
    quantity_received = models.DecimalField(max_digits=10, decimal_places=2)
    batch_number = models.CharField(max_length=50, blank=True)
    expiry_date = models.DateField(null=True, blank=True)
    
    def save(self, *args, **kwargs):
        super().save(*args, **kwargs)
        # Update the received quantity in the purchase order item
        self.purchase_order_item.received_quantity = sum(
            item.quantity_received 
            for item in self.purchase_order_item.goodsreceiptitem_set.all()
        )
        self.purchase_order_item.save()
        
        # Update product stock
        self.purchase_order_item.product.stock += self.quantity_received
        self.purchase_order_item.product.save()
    
    def __str__(self):
        return f"{self.quantity_received} of {self.purchase_order_item}"
