from django.contrib import admin
from django.utils.html import format_html
from django.urls import reverse
from django.utils.safestring import mark_safe

from .models import PurchaseOrder, PurchaseOrderItem, GoodsReceipt, GoodsReceiptItem

class PurchaseOrderItemInline(admin.TabularInline):
    model = PurchaseOrderItem
    extra = 1
    fields = ('product', 'quantity', 'unit_price', 'total_price', 'received_quantity')
    readonly_fields = ('total_price', 'received_quantity')
    
    def has_delete_permission(self, request, obj=None):
        return False

class GoodsReceiptInline(admin.TabularInline):
    model = GoodsReceipt
    extra = 0
    fields = ('grn_number', 'received_date', 'received_by', 'created_at')
    readonly_fields = ('grn_number', 'received_date', 'received_by', 'created_at')
    
    def has_add_permission(self, request, obj=None):
        return False

@admin.register(PurchaseOrder)
class PurchaseOrderAdmin(admin.ModelAdmin):
    list_display = ('po_number', 'supplier', 'order_date', 'status', 'total_amount_display', 'created_by', 'branch')
    list_filter = ('status', 'order_date', 'branch')
    search_fields = ('po_number', 'supplier__name', 'notes')
    readonly_fields = ('created_by', 'created_at', 'updated_at', 'total_amount_display')
    fieldsets = (
        ('Order Information', {
            'fields': ('po_number', 'supplier', 'order_date', 'expected_delivery_date', 'status')
        }),
        ('Details', {
            'fields': ('notes', 'branch', 'total_amount_display')
        }),
        ('Audit Information', {
            'fields': ('created_by', 'created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )
    inlines = [PurchaseOrderItemInline, GoodsReceiptInline]
    
    def save_model(self, request, obj, form, change):
        if not obj.pk:
            obj.created_by = request.user
        super().save_model(request, obj, form, change)
    
    def total_amount_display(self, obj):
        return f"${obj.total_amount:,.2f}"
    total_amount_display.short_description = 'Total Amount'

class GoodsReceiptItemInline(admin.TabularInline):
    model = GoodsReceiptItem
    extra = 0
    fields = ('purchase_order_item', 'quantity_received', 'batch_number', 'expiry_date')
    readonly_fields = ('purchase_order_item',)
    
    def has_add_permission(self, request, obj=None):
        return False

@admin.register(GoodsReceipt)
class GoodsReceiptAdmin(admin.ModelAdmin):
    list_display = ('grn_number', 'purchase_order_link', 'received_date', 'received_by', 'created_at')
    list_filter = ('received_date', 'received_by')
    search_fields = ('grn_number', 'purchase_order__po_number', 'notes')
    readonly_fields = ('created_at', 'purchase_order_link')
    inlines = [GoodsReceiptItemInline]
    
    def purchase_order_link(self, obj):
        url = reverse('admin:purchases_purchaseorder_change', args=[obj.purchase_order.id])
        return mark_safe(f'<a href="{url}">{obj.purchase_order.po_number}</a>')
    purchase_order_link.short_description = 'Purchase Order'
    purchase_order_link.admin_order_field = 'purchase_order__po_number'

@admin.register(PurchaseOrderItem)
class PurchaseOrderItemAdmin(admin.ModelAdmin):
    list_display = ('purchase_order', 'product', 'quantity', 'unit_price', 'total_price', 'received_quantity')
    list_filter = ('purchase_order__status',)
    search_fields = ('purchase_order__po_number', 'product__name', 'product__sku')
    readonly_fields = ('total_price', 'received_quantity')
    
    def has_add_permission(self, request):
        return False

@admin.register(GoodsReceiptItem)
class GoodsReceiptItemAdmin(admin.ModelAdmin):
    list_display = ('goods_receipt', 'purchase_order_item', 'quantity_received', 'batch_number', 'expiry_date')
    list_filter = ('goods_receipt__received_date',)
    search_fields = ('batch_number', 'purchase_order_item__product__name')
    
    def has_add_permission(self, request):
        return False
