import os
import json
from datetime import datetime, timedelta
from decimal import Decimal

from django.shortcuts import render, get_object_or_404, redirect
from django.views.generic import (
    ListView, DetailView, CreateView, UpdateView, DeleteView, TemplateView, FormView
)
from django.urls import reverse_lazy, reverse
from django.contrib import messages
from django.contrib.auth.mixins import LoginRequiredMixin, UserPassesTestMixin, PermissionRequiredMixin
from django.http import JsonResponse, HttpResponse, HttpResponseRedirect, Http404
from django.views.decorators.http import require_http_methods
from django.contrib.auth.decorators import login_required, permission_required
from django.utils.decorators import method_decorator
from django.views.generic.edit import FormMixin
from django.db.models import Sum, Count, Q, F, Value, DecimalField
from django.db.models.functions import Coalesce
from django.utils import timezone
from django.utils.translation import gettext_lazy as _
from django.views.decorators.csrf import csrf_exempt
from django.template.loader import render_to_string
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger

from .models import Expense, ExpenseCategory, ExpenseItem
from .forms import (
    ExpenseForm, ExpenseItemForm, ExpenseCategoryForm,
    ExpenseFilterForm, ExpenseItemFormSet
)
from branches.models import Branch


class ExpenseListView(LoginRequiredMixin, ListView):
    model = Expense
    template_name = 'expenses/expense_list.html'
    context_object_name = 'expenses'
    paginate_by = 20
    
    def get_queryset(self):
        # Get all expenses with related data
        queryset = Expense.objects.select_related('category', 'branch', 'created_by')
        
        # Debug: Print total count before any filters
        print(f"Total expenses in database: {queryset.count()}")
        print(f"Current user: {self.request.user.email}")
        
        # Get filter parameters from request
        date_from = self.request.GET.get('start_date')
        date_to = self.request.GET.get('end_date')
        status = self.request.GET.get('status')
        category_id = self.request.GET.get('category')
        search = self.request.GET.get('q')
        branch_id = self.request.GET.get('branch')
        
        # Debug: Print filter parameters
        print(f"Filters - Date: {date_from} to {date_to}, Status: {status}, Category: {category_id}, Search: {search}, Branch: {branch_id}")
        print(f"Raw GET data: {dict(self.request.GET)}")
        
        # If user is superuser, show all expenses
        if self.request.user.is_superuser:
            print("Superuser detected - showing all expenses")
        # If user has an employee profile with a branch
        elif hasattr(self.request.user, 'employee') and self.request.user.employee.branch:
            user_branch = self.request.user.employee.branch
            print(f"User's branch: {user_branch.name} (ID: {user_branch.id})")
            
            # Get all branch IDs the user has access to
            branch_ids = [user_branch.id]
            
            # Check if there are any other branches the user might have access to
            if hasattr(self.request.user, 'branches'):
                branch_ids.extend(list(self.request.user.branches.values_list('id', flat=True)))
            
            # Create a unique list of branch IDs
            branch_ids = list(set(branch_ids))
            
            # Filter by branch or show user's own expenses
            queryset = queryset.filter(
                Q(branch_id__in=branch_ids) |  # User's branch or other accessible branches
                Q(created_by=self.request.user) |  # Or expenses created by the user
                Q(branch__isnull=True)  # Or expenses with no branch
            )
            print(f"Filtered by user's branches ({branch_ids}), user's expenses, or no branch: {queryset.count()} expenses")
        else:
            # For regular users, only show their own expenses
            queryset = queryset.filter(created_by=self.request.user)
            print(f"Showing user's own expenses: {queryset.count()} expenses")
            
        # Apply status filter if provided
        if status:
            queryset = queryset.filter(status=status)
            print(f"After status filter ({status}): {queryset.count()} expenses")
            
        # Apply search filter if provided
        if search:
            queryset = queryset.filter(
                Q(reference__icontains=search) |
                Q(description__icontains=search) |
                Q(notes__icontains=search)
            )
            print(f"After search filter: {queryset.count()} expenses")
            
        # Apply category filter if provided
        if category_id:
            queryset = queryset.filter(category_id=category_id)
            print(f"After category filter: {queryset.count()} expenses")
            
        # Apply date filters
        if date_from:
            try:
                date_from = datetime.strptime(date_from, '%Y-%m-%d').date()
                queryset = queryset.filter(expense_date__gte=date_from)
                print(f"After date_from filter: {queryset.count()} expenses")
            except (ValueError, TypeError) as e:
                print(f"Error parsing date_from: {e}")
                
        if date_to:
            try:
                date_to = datetime.strptime(date_to, '%Y-%m-%d').date()
                queryset = queryset.filter(expense_date__lte=date_to)
                print(f"After date_to filter: {queryset.count()} expenses")
            except (ValueError, TypeError) as e:
                print(f"Error parsing date_to: {e}")
        
        # Order by most recent first
        queryset = queryset.order_by('-expense_date', '-created_at')
        
        # Debug: Print the final queryset SQL and sample data
        print(f"Final SQL Query: {str(queryset.query)}")
        print(f"Final queryset count: {queryset.count()}")
        
        # Print some sample data for debugging
        for expense in queryset[:5]:
            print(f"Expense: {expense.reference}, Branch: {expense.branch}, Status: {expense.status}, Created By: {expense.created_by}")
            
        return queryset
        
        # Apply date filters
        if date_from:
            try:
                date_from = datetime.strptime(date_from, '%Y-%m-%d').date()
                queryset = queryset.filter(expense_date__gte=date_from)
                print(f"After date_from filter: {queryset.count()} expenses")
            except (ValueError, TypeError) as e:
                print(f"Error parsing date_from: {e}")
                
        if date_to:
            try:
                date_to = datetime.strptime(date_to, '%Y-%m-%d').date()
                queryset = queryset.filter(expense_date__lte=date_to)
                print(f"After date_to filter: {queryset.count()} expenses")
            except (ValueError, TypeError) as e:
                print(f"Error parsing date_to: {e}")
                
        # Apply status filter - if no status is selected, show all statuses
        if status and status in ['draft', 'submitted', 'approved', 'rejected', 'paid']:
            queryset = queryset.filter(status=status)
            print(f"After status filter: {queryset.count()} expenses")
            
        # Apply category filter
            queryset = queryset.filter(category_id=int(category_id))
            print(f"After category filter: {queryset.count()} expenses")
            
        # Apply search
        if search:
            queryset = queryset.filter(
                Q(reference__icontains=search) |
                Q(description__icontains=search) |
                Q(payment_reference__icontains=search)
            )
            print(f"After search filter: {queryset.count()} expenses")
        
        # TEMPORARILY DISABLE PERMISSION CHECKS
        # if not self.request.user.has_perm('expenses.view_all_expenses'):
        #     # Only show expenses created by the user or for their branch
        #     if hasattr(self.request.user, 'employee') and self.request.user.employee.branch:
        #         queryset = queryset.filter(
        #             Q(created_by=self.request.user) |
        #             Q(branch=self.request.user.employee.branch)
        #         )
        #     else:
        #         queryset = queryset.filter(created_by=self.request.user)
        
        # Order by date (newest first) and creation time
        queryset = queryset.order_by('-expense_date', '-created_at')
        print(f"Final queryset count: {queryset.count()}")
        
        return queryset
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        
        # Add categories to context
        context['categories'] = ExpenseCategory.objects.filter(is_active=True).order_by('name')
        
        # Add summary statistics
        queryset = self.get_queryset()
        context['total_expenses'] = queryset.count()
        context['total_amount'] = queryset.aggregate(
            total=Coalesce(Sum('total_amount'), Decimal('0.00'), output_field=DecimalField())
        )['total']
        
        # Add status counts for filter
        context['status_counts'] = queryset.values('status').annotate(
            count=Count('id'),
            total=Coalesce(Sum('total_amount'), Decimal('0.00'), output_field=DecimalField())
        ).order_by('status')
        
        # Add recent expenses
        context['recent_expenses'] = queryset[:5]
        
        # Add permissions
        context['can_add_expense'] = self.request.user.has_perm('expenses.add_expense')
        context['can_approve_expense'] = self.request.user.has_perm('expenses.approve_expense')
        
        return context


class ExpenseDetailView(LoginRequiredMixin, DetailView):
    model = Expense
    template_name = 'expenses/expense_detail.html'
    context_object_name = 'expense'
    
    def get_queryset(self):
        queryset = super().get_queryset()
        queryset = queryset.select_related('category', 'branch', 'created_by', 'approved_by')
        return queryset
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        expense = self.get_object()
        
        # Add related items
        context['items'] = expense.items.all()
        
        # Add permissions
        context['can_edit'] = self.request.user.has_perm('expenses.change_expense')
        context['can_delete'] = self.request.user.has_perm('expenses.delete_expense')
        context['can_approve'] = (
            self.request.user.has_perm('expenses.approve_expense') and
            expense.status in ['draft', 'pending']
        )
        
        # Add action buttons
        context['show_approve_btn'] = (
            expense.status in ['draft', 'pending'] and
            self.request.user.has_perm('expenses.approve_expense')
        )
        
        context['show_pay_btn'] = (
            expense.status == 'approved' and
            self.request.user.has_perm('expenses.change_expense')
        )
        
        return context


class ExpenseCreateView(LoginRequiredMixin, CreateView):
    model = Expense
    form_class = ExpenseForm
    template_name = 'expenses/expense_form.html'
    
    def get_form_kwargs(self):
        kwargs = super().get_form_kwargs()
        kwargs['request'] = self.request
        return kwargs
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        
        # Add default category if available
        default_category = ExpenseCategory.objects.filter(is_active=True).first()
        if default_category and 'category' not in context['form'].initial:
            context['form'].initial['category'] = default_category
        
        return context
    
    def form_valid(self, form):
        try:
            # Set the created_by field to the current user
            form.instance.created_by = self.request.user
            
            # Debug: Print user and branch info
            print(f"Creating expense for user: {self.request.user.email}")
            
            # Set the branch if user has an employee profile with a branch
            if hasattr(self.request.user, 'employee') and self.request.user.employee.branch:
                form.instance.branch = self.request.user.employee.branch
                print(f"Set branch to: {form.instance.branch}")
            else:
                print("No branch found for user")
            
            # Save the expense
            self.object = form.save()
            print(f"Expense created with ID: {self.object.id}, Reference: {self.object.reference}")
            
            # Set amount as total_amount for compatibility
            self.object.total_amount = self.object.amount
            self.object.save(update_fields=['total_amount'])
            
            # Debug: Print the saved expense details
            print(f"Expense details - Branch: {self.object.branch}, Status: {self.object.status}, Amount: {self.object.amount}")
            
            # Add success message
            messages.success(
                self.request,
                f'Expense {self.object.reference} has been created successfully.'
            )
            
            # Redirect to expenses list with success message
            return redirect('expenses:expense_list')
            
        except Exception as e:
            # Log the error for debugging
            import traceback
            print(f"Error saving expense: {str(e)}")
            traceback.print_exc()
            
            # Add error message
            messages.error(
                self.request,
                f'An error occurred while saving the expense: {str(e)}. Please try again.'
            )
            
            # Return form with errors
            return self.form_invalid(form)


class ExpenseUpdateView(LoginRequiredMixin, UpdateView):
    model = Expense
    form_class = ExpenseForm
    template_name = 'expenses/expense_form.html'
    
    def get_form_kwargs(self):
        kwargs = super().get_form_kwargs()
        kwargs['request'] = self.request
        return kwargs
    
    def get_queryset(self):
        queryset = super().get_queryset()
        # Only allow updating expenses created by the user or if user has permission
        if not self.request.user.is_superuser:
            queryset = queryset.filter(created_by=self.request.user)
        return queryset
    
    def form_valid(self, form):
        # Save the expense
        self.object = form.save()
        
        # Update total_amount to match amount for compatibility
        self.object.total_amount = self.object.amount
        self.object.save(update_fields=['total_amount'])
        
        messages.success(
            self.request,
            _('Expense has been updated successfully.')
        )
        
        return redirect('expenses:expense_detail', pk=self.object.pk)


class ExpenseDeleteView(LoginRequiredMixin, DeleteView):
    model = Expense
    template_name = 'expenses/expense_confirm_delete.html'
    success_url = reverse_lazy('expenses:expense_list')
    
    def get_queryset(self):
        queryset = super().get_queryset()
        
        # Only allow deletion of draft or pending expenses
        if not self.request.user.is_superuser:
            queryset = queryset.filter(status__in=['draft', 'pending'])
        
        return queryset
    
    def delete(self, request, *args, **kwargs):
        self.object = self.get_object()
        success_url = self.get_success_url()
        
        # Delete the expense and its items
        self.object.delete()
        
        messages.success(
            request,
            _('Expense has been deleted successfully.')
        )
        
        return HttpResponseRedirect(success_url)


@method_decorator(login_required, name='dispatch')
class ExpenseApproveView(LoginRequiredMixin, UserPassesTestMixin, UpdateView):
    model = Expense
    fields = []
    template_name = 'expenses/expense_approve.html'
    
    def test_func(self):
        expense = self.get_object()
        return expense.status == 'submitted' and self.request.user.has_perm('expenses.approve_expense')
    
    def get_queryset(self):
        return Expense.objects.filter(branch=self.request.user.branch)
    
    def form_valid(self, form):
        expense = form.save(commit=False)
        expense.status = 'approved'
        expense.approved_by = self.request.user
        expense.approved_at = timezone.now()
        expense.save()
        messages.success(self.request, f'Expense #{expense.reference} has been approved.')
        return super().form_valid(form)
    
    def get_success_url(self):
        return reverse('expenses:expense_detail', kwargs={'pk': self.object.pk})


class ExpenseRejectView(LoginRequiredMixin, UserPassesTestMixin, UpdateView):
    model = Expense
    fields = ['rejection_reason']
    template_name = 'expenses/expense_reject.html'
    
    def test_func(self):
        expense = self.get_object()
        return expense.status == 'submitted' and self.request.user.has_perm('expenses.approve_expense')
    
    def get_queryset(self):
        return Expense.objects.filter(branch=self.request.user.branch)
    
    def form_valid(self, form):
        expense = form.save(commit=False)
        expense.status = 'rejected'
        expense.rejected_by = self.request.user
        expense.rejected_at = timezone.now()
        expense.save()
        messages.warning(self.request, f'Expense #{expense.reference} has been rejected.')
        return super().form_valid(form)
    
    def get_success_url(self):
        return reverse('expenses:expense_detail', kwargs={'pk': self.object.pk})


@method_decorator(login_required, name='dispatch')
class ExpensePayView(UserPassesTestMixin, UpdateView):
    model = Expense
    fields = ['payment_date', 'payment_method', 'payment_reference']
    template_name = 'expenses/expense_pay.html'
    
    def test_func(self):
        return self.request.user.has_perm('expenses.change_expense')
    
    def get_queryset(self):
        return super().get_queryset().filter(status='approved')
    
    def get_form(self, form_class=None):
        form = super().get_form(form_class)
        
        # Set default payment date to today
        if 'payment_date' in form.fields:
            form.fields['payment_date'].initial = timezone.now().date()
        
        return form
    
    def form_valid(self, form):
        self.object = form.save(commit=False)
        self.object.status = 'paid'
        self.object.payment_date = form.cleaned_data['payment_date']
        self.object.save(update_fields=['status', 'payment_date'])
        
        messages.success(
            self.request,
            _('Expense has been marked as paid successfully.')
        )
        
        return redirect('expenses:expense_detail', pk=self.object.pk)


class ExpenseCategoryListView(LoginRequiredMixin, ListView):
    model = ExpenseCategory
    template_name = 'expenses/expensecategory_list.html'
    context_object_name = 'categories'
    
    def get_queryset(self):
        queryset = super().get_queryset()
        
        # Filter by active status if specified
        active = self.request.GET.get('active')
        if active == 'true':
            queryset = queryset.filter(is_active=True)
        elif active == 'false':
            queryset = queryset.filter(is_active=False)
        
        return queryset.order_by('name')


class ExpenseCategoryCreateView(LoginRequiredMixin, CreateView):
    model = ExpenseCategory
    form_class = ExpenseCategoryForm
    template_name = 'expenses/expensecategory_form.html'
    success_url = reverse_lazy('expenses:expensecategory_list')
    
    def form_valid(self, form):
        messages.success(
            self.request,
            _('Expense category has been created successfully.')
        )
        return super().form_valid(form)


class ExpenseCategoryUpdateView(LoginRequiredMixin, UpdateView):
    model = ExpenseCategory
    form_class = ExpenseCategoryForm
    template_name = 'expenses/expensecategory_form.html'
    success_url = reverse_lazy('expenses:expensecategory_list')
    
    def form_valid(self, form):
        messages.success(
            self.request,
            _('Expense category has been updated successfully.')
        )
        return super().form_valid(form)


class ExpenseCategoryDeleteView(LoginRequiredMixin, DeleteView):
    model = ExpenseCategory
    template_name = 'expenses/expensecategory_confirm_delete.html'
    success_url = reverse_lazy('expenses:expensecategory_list')
    
    def delete(self, request, *args, **kwargs):
        self.object = self.get_object()
        success_url = self.get_success_url()
        
        # Check if category is in use
        if self.object.expenses.exists():
            messages.error(
                request,
                _('Cannot delete this category because it is in use by one or more expenses.')
            )
            return redirect(success_url)
        
        self.object.delete()
        
        messages.success(
            request,
            _('Expense category has been deleted successfully.')
        )
        
        return HttpResponseRedirect(success_url)


@login_required
def expense_dashboard(request):
    """Expense dashboard with charts and statistics"""
    # Date ranges
    today = timezone.now().date()
    last_month = today - timedelta(days=30)
    
    # Get base queryset
    expenses = Expense.objects.all()
    
    # Apply user permissions
    if not request.user.has_perm('expenses.view_all_expenses'):
        if hasattr(request.user, 'employee') and request.user.employee.branch:
            expenses = expenses.filter(
                Q(created_by=request.user) |
                Q(branch=request.user.employee.branch)
            )
        else:
            expenses = expenses.filter(created_by=request.user)
    
    # Recent expenses
    recent_expenses = expenses.order_by('-expense_date')[:5]
    
    # Monthly summary
    monthly_summary = expenses.filter(
        expense_date__gte=last_month
    ).values(
        'expense_date__year', 'expense_date__month'
    ).annotate(
        month=models.functions.TruncMonth('expense_date'),
        total=Sum('total_amount'),
        count=Count('id')
    ).order_by('month')
    
    # Category summary
    category_summary = expenses.values(
        'category__name'
    ).annotate(
        total=Sum('total_amount'),
        count=Count('id')
    ).order_by('-total')[:10]
    
    # Status summary
    status_summary = expenses.values(
        'status'
    ).annotate(
        total=Sum('total_amount'),
        count=Count('id')
    ).order_by('status')
    
    context = {
        'recent_expenses': recent_expenses,
        'monthly_summary': monthly_summary,
        'category_summary': category_summary,
        'status_summary': status_summary,
        'can_add_expense': request.user.has_perm('expenses.add_expense'),
        'can_approve_expense': request.user.has_perm('expenses.approve_expense'),
    }
    
    return render(request, 'expenses/expense_dashboard.html', context)


@login_required
def export_expenses(request, format='csv'):
    """Export expenses to CSV or Excel"""
    # Check permission
    if not request.user.has_perm('expenses.export_expenses'):
        raise Http404
    
    # Get filtered expenses
    expenses = Expense.objects.select_related('category', 'branch', 'created_by')
    
    # Apply filters from GET parameters
    form = ExpenseFilterForm(request.GET, request=request)
    if form.is_valid():
        date_from = form.cleaned_data.get('date_from')
        date_to = form.cleaned_data.get('date_to')
        status = form.cleaned_data.get('status')
        category = form.cleaned_data.get('category')
        branch = form.cleaned_data.get('branch')
        search = form.cleaned_data.get('search')
        
        if date_from:
            expenses = expenses.filter(expense_date__gte=date_from)
        if date_to:
            expenses = expenses.filter(expense_date__lte=date_to)
        if status:
            expenses = expenses.filter(status=status)
        if category:
            expenses = expenses.filter(category=category)
        if branch:
            expenses = expenses.filter(branch=branch)
        if search:
            expenses = expenses.filter(
                Q(reference__icontains=search) |
                Q(description__icontains=search) |
                Q(payment_reference__icontains=search)
            )
    
    # Order the results
    expenses = expenses.order_by('-expense_date', '-created_at')
    
    # Prepare data for export
    data = []
    
    # Add headers
    headers = [
        _('Reference'), _('Date'), _('Category'), _('Branch'),
        _('Description'), _('Amount'), _('Tax'), _('Total'),
        _('Status'), _('Payment Method'), _('Payment Date'), _('Payment Reference'),
        _('Created By'), _('Created At'), _('Approved By'), _('Approved At')
    ]
    data.append(headers)
    
    # Add data rows
    for expense in expenses:
        data.append([
            expense.reference,
            expense.expense_date.strftime('%Y-%m-%d') if expense.expense_date else '',
            str(expense.category) if expense.category else '',
            str(expense.branch) if expense.branch else '',
            expense.description or '',
            str(expense.amount or '0.00'),
            str(expense.tax_amount or '0.00'),
            str(expense.total_amount or '0.00'),
            expense.get_status_display(),
            expense.get_payment_method_display(),
            expense.payment_date.strftime('%Y-%m-%d') if expense.payment_date else '',
            expense.payment_reference or '',
            str(expense.created_by) if expense.created_by else '',
            expense.created_at.strftime('%Y-%m-%d %H:%M') if expense.created_at else '',
            str(expense.approved_by) if expense.approved_by else '',
            expense.approved_at.strftime('%Y-%m-%d %H:%M') if expense.approved_at else ''
        ])
    
    # Create response based on format
    if format == 'csv':
        import csv
        from django.http import HttpResponse
        
        response = HttpResponse(content_type='text/csv')
        response['Content-Disposition'] = 'attachment; filename="expenses_export.csv"'
        
        writer = csv.writer(response)
        writer.writerows(data)
        
        return response
    
    elif format == 'excel':
        import openpyxl
        from openpyxl.writer.excel import save_virtual_workbook
        from django.http import HttpResponse
        
        wb = openpyxl.Workbook()
        ws = wb.active
        ws.title = 'Expenses'
        
        # Add data to worksheet
        for row in data:
            ws.append(row)
        
        # Style the header row
        from openpyxl.styles import Font
        for cell in ws[1]:
            cell.font = Font(bold=True)
        
        # Auto-size columns
        from openpyxl.utils import get_column_letter
        for i, column in enumerate(ws.columns, 1):
            max_length = 0
            column_letter = get_column_letter(i)
            
            for cell in column:
                try:
                    if len(str(cell.value)) > max_length:
                        max_length = len(str(cell.value))
                except:
                    pass
            
            adjusted_width = (max_length + 2) * 1.2
            ws.column_dimensions[column_letter].width = min(adjusted_width, 50)
        
        # Save the workbook to a response
        response = HttpResponse(
            save_virtual_workbook(wb),
            content_type='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
        )
        response['Content-Disposition'] = 'attachment; filename="expenses_export.xlsx"'
        
        return response
    
    else:
        # Default to JSON if format is not recognized
        import json
        from django.http import JsonResponse
        
        # Convert data to list of dicts
        headers = data[0]
        result = []
        
        for row in data[1:]:
            result.append(dict(zip(headers, row)))
        
        return JsonResponse(result, safe=False)


@login_required
def expense_autocomplete(request):
    """AJAX endpoint for expense autocomplete"""
    query = request.GET.get('q', '').strip()
    
    if not query:
        return JsonResponse({'results': []})
    
    # Get base queryset
    expenses = Expense.objects.select_related('category', 'branch')
    
    # Apply user permissions
    if not request.user.has_perm('expenses.view_all_expenses'):
        if hasattr(request.user, 'employee') and request.user.employee.branch:
            expenses = expenses.filter(
                Q(created_by=request.user) |
                Q(branch=request.user.employee.branch)
            )
        else:
            expenses = expenses.filter(created_by=request.user)
    
    # Apply search
    expenses = expenses.filter(
        Q(reference__icontains=query) |
        Q(description__icontains=query) |
        Q(payment_reference__icontains=query)
    ).order_by('-expense_date')[:10]
    
    # Prepare results
    results = []
    for expense in expenses:
        results.append({
            'id': expense.id,
            'text': f"{expense.reference} - {expense.description or 'No description'}",
            'reference': expense.reference,
            'description': expense.description or '',
            'amount': str(expense.total_amount or '0.00'),
            'date': expense.expense_date.strftime('%Y-%m-%d') if expense.expense_date else '',
            'status': expense.get_status_display(),
            'status_value': expense.status,
            'category': str(expense.category) if expense.category else '',
            'branch': str(expense.branch) if expense.branch else '',
            'url': reverse('expenses:expense_detail', kwargs={'pk': expense.pk})
        })
    
    return JsonResponse({'results': results})
