from django import forms
from django.forms import ModelForm, inlineformset_factory
from django.utils.translation import gettext_lazy as _
from django.core.validators import MinValueValidator, FileExtensionValidator
from django.utils import timezone
from decimal import Decimal
import os

from .models import Expense, ExpenseCategory, ExpenseItem
from branches.models import Branch

class ExpenseCategoryForm(forms.ModelForm):
    class Meta:
        model = ExpenseCategory
        fields = ['name', 'description', 'is_active']
        widgets = {
            'name': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Enter category name')
            }),
            'description': forms.Textarea(attrs={
                'class': 'form-control',
                'rows': 3,
                'placeholder': _('Enter a brief description')
            }),
            'is_active': forms.CheckboxInput(attrs={
                'class': 'form-check-input'
            })
        }
        labels = {
            'name': _('Category Name'),
            'is_active': _('Active')
        }

class ExpenseForm(forms.ModelForm):
    new_category = forms.CharField(
        required=False,
        widget=forms.TextInput(attrs={
            'class': 'form-control',
            'placeholder': _('Enter new category name')
        }),
        label=_('New Category')
    )
    
    class Meta:
        model = Expense
        fields = ['description', 'category', 'payment_method', 'amount']
        widgets = {
            'description': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Expense name/description')
            }),
            'category': forms.Select(attrs={
                'class': 'form-select',
                'data-control': 'select2',
                'data-placeholder': _('Select a category')
            }),
            'payment_method': forms.Select(attrs={
                'class': 'form-select',
                'data-control': 'select2'
            }),
            'amount': forms.NumberInput(attrs={
                'class': 'form-control',
                'step': '0.01',
                'min': '0.01',
                'placeholder': '0.00'
            })
        }
        help_texts = {
            'receipt': _('Supported formats: JPG, PNG, PDF (Max 5MB)'),
        }

    def __init__(self, *args, **kwargs):
        self.request = kwargs.pop('request', None)
        super().__init__(*args, **kwargs)
        
        # Set default amount
        if not self.instance.pk:
            self.fields['amount'].initial = 0.00
        
        # Set category choices to include a 'New Category' option
        self.fields['category'].choices = [
            ('', '---------'),
            ('new', '➕ Add New Category')
        ] + list(ExpenseCategory.objects.filter(is_active=True).values_list('id', 'name'))
        
        # Set payment method choices
        self.fields['payment_method'].choices = [
            ('', '---------'),
            ('cash', 'Cash'),
            ('bank_transfer', 'Bank Transfer'),
            ('credit_card', 'Credit Card'),
            ('mobile_money', 'Mobile Money'),
            ('other', 'Other')
        ]
        
        # Make fields required
        for field in self.fields:
            self.fields[field].required = True
    
    def clean(self):
        cleaned_data = super().clean()
        category = cleaned_data.get('category')
        new_category = cleaned_data.get('new_category')
        
        # If 'Add New Category' is selected but no category name is provided
        if category == 'new' and not new_category:
            self.add_error('new_category', 'Please enter a category name')
        
        return cleaned_data
    
    def save(self, commit=True):
        expense = super().save(commit=False)
        
        # Set default values
        if not expense.pk:
            expense.status = 'approved'  # Auto-approve simple expenses
            expense.expense_date = timezone.now().date()
            
            # Set branch to user's branch if available
            if hasattr(self.request.user, 'employee') and self.request.user.employee.branch:
                expense.branch = self.request.user.employee.branch
        
        # Handle new category creation
        if self.cleaned_data.get('category') == 'new' and self.cleaned_data.get('new_category'):
            category_name = self.cleaned_data['new_category'].strip()
            # Check if category already exists (case-insensitive)
            existing_category = ExpenseCategory.objects.filter(name__iexact=category_name).first()
            
            if existing_category:
                # Use existing category
                expense.category = existing_category
            else:
                # Create new category
                category = ExpenseCategory.objects.create(
                    name=category_name,
                    description=f'Created from expense on {timezone.now().strftime("%Y-%m-%d")}',
                    is_active=True
                )
                expense.category = category
        
        # Set created_by if this is a new expense
        if not expense.pk and hasattr(self, 'request'):
            expense.created_by = self.request.user
        
        if commit:
            expense.save()
            
        return expense
    
    def clean_receipt(self):
        receipt = self.cleaned_data.get('receipt')
        if receipt:
            # Check file size (5MB max)
            max_size = 5 * 1024 * 1024  # 5MB
            if receipt.size > max_size:
                raise forms.ValidationError(_('File size must be no more than 5MB.'))
            
            # Check file extension
            valid_extensions = ['.jpg', '.jpeg', '.png', '.pdf']
            ext = os.path.splitext(receipt.name)[1].lower()
            if ext not in valid_extensions:
                raise forms.ValidationError(_('Unsupported file type. Supported formats: JPG, PNG, PDF'))
        
        return receipt
    
    def clean(self):
        cleaned_data = super().clean()
        amount = cleaned_data.get('amount', 0) or 0
        tax_amount = cleaned_data.get('tax_amount', 0) or 0
        total_amount = cleaned_data.get('total_amount', 0) or 0
        
        # Validate total amount
        if amount and tax_amount and total_amount != amount + tax_amount:
            self.add_error('total_amount', _('Total amount must be the sum of amount and tax amount.'))
        
        return cleaned_data


class ExpenseItemForm(forms.ModelForm):
    class Meta:
        model = ExpenseItem
        fields = ['description', 'quantity', 'unit_price', 'tax_rate', 'tax_amount', 'amount', 'total_amount']
        widgets = {
            'description': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': _('Item description')
            }),
            'quantity': forms.NumberInput(attrs={
                'class': 'form-control',
                'step': '0.01',
                'min': '0.01',
                'data-quantity': 'true',
            }),
            'unit_price': forms.NumberInput(attrs={
                'class': 'form-control',
                'step': '0.01',
                'min': '0.01',
                'data-unit-price': 'true',
            }),
            'tax_rate': forms.NumberInput(attrs={
                'class': 'form-control',
                'step': '0.01',
                'min': '0',
                'max': '100',
                'data-tax-rate': 'true',
            }),
            'tax_amount': forms.NumberInput(attrs={
                'class': 'form-control',
                'readonly': True,
                'data-tax-amount': 'true',
            }),
            'amount': forms.NumberInput(attrs={
                'class': 'form-control',
                'readonly': True,
                'data-amount': 'true',
            }),
            'total_amount': forms.NumberInput(attrs={
                'class': 'form-control',
                'readonly': True,
                'data-total-amount': 'true',
            }),
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        # Set default tax rate if not set
        if not self.instance.pk:
            self.fields['tax_rate'].initial = 0.00
            self.fields['quantity'].initial = 1.00


# Formset for expense items
ExpenseItemFormSet = inlineformset_factory(
    Expense,
    ExpenseItem,
    form=ExpenseItemForm,
    extra=1,
    can_delete=True,
    min_num=1,
    validate_min=True
)


class ExpenseFilterForm(forms.Form):
    STATUS_CHOICES = [
        ('', _('All Status')),
        ('draft', _('Draft')),
        ('pending', _('Pending Approval')),
        ('approved', _('Approved')),
        ('rejected', _('Rejected')),
        ('paid', _('Paid')),
    ]
    
    date_from = forms.DateField(
        required=False,
        widget=forms.DateInput(attrs={
            'class': 'form-control',
            'type': 'date',
            'data-provider': 'flatpickr',
            'data-date-format': 'Y-m-d',
            'placeholder': _('From date')
        })
    )
    
    date_to = forms.DateField(
        required=False,
        widget=forms.DateInput(attrs={
            'class': 'form-control',
            'type': 'date',
            'data-provider': 'flatpickr',
            'data-date-format': 'Y-m-d',
            'placeholder': _('To date')
        })
    )
    
    status = forms.ChoiceField(
        required=False,
        choices=STATUS_CHOICES,
        widget=forms.Select(attrs={
            'class': 'form-select',
        })
    )
    
    category = forms.ModelChoiceField(
        required=False,
        queryset=ExpenseCategory.objects.filter(is_active=True),
        widget=forms.Select(attrs={
            'class': 'form-select',
        })
    )
    
    branch = forms.ModelChoiceField(
        required=False,
        queryset=Branch.objects.filter(is_active=True),
        widget=forms.Select(attrs={
            'class': 'form-select',
        })
    )
    
    search = forms.CharField(
        required=False,
        widget=forms.TextInput(attrs={
            'class': 'form-control',
            'placeholder': _('Search by reference or description...')
        })
    )
    
    def __init__(self, *args, **kwargs):
        self.request = kwargs.pop('request', None)
        super().__init__(*args, **kwargs)
        
        # Limit branches to user's accessible branches
        if hasattr(self.request.user, 'employee') and not self.request.user.is_superuser:
            self.fields['branch'].queryset = Branch.objects.filter(
                id=self.request.user.employee.branch_id
            )
