from django import forms
from .models import Category, Product

class CategoryForm(forms.ModelForm):
    """
    Form for category management
    """
    class Meta:
        model = Category
        fields = ['name', 'description', 'branch']
        widgets = {
            'name': forms.TextInput(attrs={'class': 'form-control'}),
            'description': forms.Textarea(attrs={'class': 'form-control', 'rows': 3}),
            'branch': forms.Select(attrs={'class': 'form-control'}),
        }

class ProductForm(forms.ModelForm):
    """
    Form for product management
    """
    # Define cost_price as a field with required=False
    cost_price = forms.DecimalField(
        max_digits=10, 
        decimal_places=2, 
        required=False, 
        initial=0,
        widget=forms.NumberInput(attrs={'class': 'form-control', 'step': '0.01'})
    )
    
    class Meta:
        model = Product
        fields = [
            'name', 'description', 'category', 'branch', 'price', 
            'cost_price', 'quantity', 'low_stock_threshold', 
            'sku', 'barcode', 'image', 'is_active'
        ]
        widgets = {
            'name': forms.TextInput(attrs={'class': 'form-control'}),
            'description': forms.Textarea(attrs={'class': 'form-control', 'rows': 3}),
            'category': forms.Select(attrs={'class': 'form-control'}),
            'branch': forms.Select(attrs={'class': 'form-control'}),
            'price': forms.NumberInput(attrs={'class': 'form-control', 'step': '0.01'}),
            'quantity': forms.NumberInput(attrs={'class': 'form-control'}),
            'low_stock_threshold': forms.NumberInput(attrs={'class': 'form-control'}),
            'sku': forms.TextInput(attrs={'class': 'form-control'}),
            'barcode': forms.TextInput(attrs={'class': 'form-control'}),
            'image': forms.FileInput(attrs={'class': 'form-control', 'accept': 'image/*'}),
            'is_active': forms.CheckboxInput(attrs={'class': 'form-check-input'}),
        }
        
    def clean_cost_price(self):
        """Ensure cost_price has a default value of 0"""
        cost_price = self.cleaned_data.get('cost_price')
        if cost_price is None:
            return 0
        return cost_price

class ProductSearchForm(forms.Form):
    """
    Form for searching products
    """
    query = forms.CharField(
        required=False,
        widget=forms.TextInput(attrs={
            'class': 'form-control',
            'placeholder': 'Search by name, SKU, or barcode'
        })
    )
    category = forms.ModelChoiceField(
        queryset=Category.objects.none(),
        required=False,
        empty_label="All Categories",
        widget=forms.Select(attrs={'class': 'form-control'})
    )
    
    def __init__(self, user, *args, **kwargs):
        super().__init__(*args, **kwargs)
        
        # Filter categories by user's branch or show all for admin
        if user.role == 'admin':
            self.fields['category'].queryset = Category.objects.all()
        else:
            self.fields['category'].queryset = Category.objects.filter(branch=user.branch)

class StockAdjustmentForm(forms.Form):
    """
    Form for adjusting product stock
    """
    ADJUSTMENT_TYPES = [
        ('add', 'Add to Stock'),
        ('remove', 'Remove from Stock'),
        ('set', 'Set Stock Level'),
    ]
    
    adjustment_type = forms.ChoiceField(
        choices=ADJUSTMENT_TYPES,
        widget=forms.Select(attrs={'class': 'form-control'})
    )
    quantity = forms.IntegerField(
        widget=forms.NumberInput(attrs={'class': 'form-control', 'min': '0'})
    )
    reason = forms.CharField(
        widget=forms.Textarea(attrs={'class': 'form-control', 'rows': 3}),
        help_text="Provide a reason for this stock adjustment."
    )
