from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required, user_passes_test
from django.contrib import messages
from django.http import JsonResponse
from django.db.models import Q

from .models import Customer
from .forms import CustomerForm
from core.models import AuditLog

def is_admin_or_manager(user):
    """
    Check if user is an admin or manager
    """
    return user.is_authenticated and (user.role in ['admin', 'manager'])

def can_manage_customers(user):
    """
    Check if user can manage customers (admin, manager, or cashier)
    """
    return user.is_authenticated and (user.role in ['admin', 'manager', 'cashier'])

@login_required
def customer_list(request):
    """
    List all customers with search functionality
    """
    # Base queryset filtered by branch if not admin
    if request.user.role == 'admin':
        customers = Customer.objects.all().select_related('branch')
    else:
        customers = Customer.objects.filter(branch=request.user.branch).select_related('branch')
    
    # Handle search
    search_query = request.GET.get('q', '')
    if search_query:
        customers = customers.filter(
            Q(first_name__icontains=search_query) | 
            Q(last_name__icontains=search_query) | 
            Q(email__icontains=search_query) | 
            Q(phone__icontains=search_query)
        )
    
    # Order by name
    customers = customers.order_by('first_name', 'last_name')
    
    context = {
        'customers': customers,
        'search_query': search_query,
    }
    return render(request, 'customers/customer_list.html', context)

@login_required
def customer_detail(request, customer_id):
    """
    Show customer details
    """
    # Get customer, ensuring branch access
    if request.user.role == 'admin':
        customer = get_object_or_404(Customer, id=customer_id)
    else:
        customer = get_object_or_404(Customer, id=customer_id, branch=request.user.branch)
    
    # Get customer's recent sales
    sales = customer.sale_set.all().order_by('-created_at')[:10]
    
    context = {
        'customer': customer,
        'sales': sales,
    }
    return render(request, 'customers/customer_detail.html', context)

@login_required
@user_passes_test(can_manage_customers)
def customer_add(request):
    """
    Add a new customer
    """
    if request.method == 'POST':
        form = CustomerForm(request.POST)
        if form.is_valid():
            customer = form.save()
            
            # Log the action
            AuditLog.log_action(
                user=request.user,
                action='create',
                entity_type='customer',
                entity_id=customer.id,
                details=f"Created customer: {customer.full_name}",
                ip_address=request.META.get('REMOTE_ADDR')
            )
            
            messages.success(request, f'Customer "{customer.full_name}" added successfully.')
            return redirect('customer_list')
    else:
        # Pre-select user's branch if not admin
        initial = {}
        if request.user.role != 'admin' and request.user.branch:
            initial['branch'] = request.user.branch
        
        form = CustomerForm(initial=initial)
        
        # Restrict branch selection if not admin
        if request.user.role != 'admin':
            form.fields['branch'].queryset = form.fields['branch'].queryset.filter(id=request.user.branch.id)
    
    context = {
        'form': form,
        'title': 'Add Customer',
    }
    return render(request, 'customers/customer_form.html', context)

@login_required
@user_passes_test(is_admin_or_manager)
def customer_edit(request, customer_id):
    """
    Edit an existing customer
    """
    # Get customer, ensuring branch access
    if request.user.role == 'admin':
        customer = get_object_or_404(Customer, id=customer_id)
    else:
        customer = get_object_or_404(Customer, id=customer_id, branch=request.user.branch)
    
    if request.method == 'POST':
        form = CustomerForm(request.POST, instance=customer)
        if form.is_valid():
            customer = form.save()
            
            # Log the action
            AuditLog.log_action(
                user=request.user,
                action='update',
                entity_type='customer',
                entity_id=customer.id,
                details=f"Updated customer: {customer.full_name}",
                ip_address=request.META.get('REMOTE_ADDR')
            )
            
            messages.success(request, f'Customer "{customer.full_name}" updated successfully.')
            return redirect('customer_list')
    else:
        form = CustomerForm(instance=customer)
        
        # Restrict branch selection if not admin
        if request.user.role != 'admin':
            form.fields['branch'].queryset = form.fields['branch'].queryset.filter(id=request.user.branch.id)
    
    context = {
        'form': form,
        'title': 'Edit Customer',
        'customer': customer,
    }
    return render(request, 'customers/customer_form.html', context)

@login_required
@user_passes_test(is_admin_or_manager)
def customer_delete(request, customer_id):
    """
    Delete a customer
    """
    # Get customer, ensuring branch access
    if request.user.role == 'admin':
        customer = get_object_or_404(Customer, id=customer_id)
    else:
        customer = get_object_or_404(Customer, id=customer_id, branch=request.user.branch)
    
    # Check if customer has sales
    if customer.sale_set.exists():
        messages.error(request, f'Cannot delete customer "{customer.full_name}" because they have associated sales.')
        return redirect('customer_list')
    
    if request.method == 'POST':
        customer_name = customer.full_name
        customer.delete()
        
        # Log the action
        AuditLog.log_action(
            user=request.user,
            action='delete',
            entity_type='customer',
            entity_id=customer_id,
            details=f"Deleted customer: {customer_name}",
            ip_address=request.META.get('REMOTE_ADDR')
        )
        
        messages.success(request, f'Customer "{customer_name}" has been deleted.')
        return redirect('customer_list')
    
    context = {
        'customer': customer,
    }
    return render(request, 'customers/customer_confirm_delete.html', context)

def customer_list_api(request):
    """
    API endpoint for getting customers, used in sale creation
    Ajax API that accepts requests with proper CSRF token
    """
    # Check if user is authenticated
    if not request.user.is_authenticated:
        return JsonResponse({'error': 'Authentication required'}, status=403)
        
    # Base queryset filtered by branch if not admin
    if request.user.role == 'admin':
        customers = Customer.objects.all()
    else:
        customers = Customer.objects.filter(branch=request.user.branch)
    
    # Apply search if provided
    search = request.GET.get('q', '')
    if search:
        customers = customers.filter(
            Q(first_name__icontains=search) | 
            Q(last_name__icontains=search) | 
            Q(phone__icontains=search)
        )
    
    # Limit results and include only necessary fields
    customers = customers[:10].values('id', 'first_name', 'last_name', 'phone', 'email')
    
    return JsonResponse({'customers': list(customers)})

def customer_create_api(request):
    """
    API endpoint for creating a customer during sale
    Ajax API that accepts requests with proper CSRF token
    """
    # Check if user is authenticated
    if not request.user.is_authenticated:
        return JsonResponse({'error': 'Authentication required'}, status=403)
    
    if request.method != 'POST':
        return JsonResponse({'success': False, 'message': 'Invalid request method'})
    
    form = CustomerForm(request.POST)
    if form.is_valid():
        customer = form.save()
        
        # Log the action
        AuditLog.log_action(
            user=request.user,
            action='create',
            entity_type='customer',
            entity_id=customer.id,
            details=f"Created customer via API: {customer.full_name}",
            ip_address=request.META.get('REMOTE_ADDR')
        )
        
        return JsonResponse({
            'success': True,
            'customer': {
                'id': customer.id,
                'first_name': customer.first_name,
                'last_name': customer.last_name,
                'email': customer.email,
                'phone': customer.phone
            }
        })
    else:
        return JsonResponse({
            'success': False,
            'message': 'Validation error',
            'errors': form.errors
        })