from django.contrib import admin
from django.utils.translation import gettext_lazy as _
from .models import Expense, ExpenseCategory

@admin.register(ExpenseCategory)
class ExpenseCategoryAdmin(admin.ModelAdmin):
    list_display = ('name', 'is_active', 'created_at', 'updated_at')
    list_filter = ('is_active',)
    search_fields = ('name', 'description')
    readonly_fields = ('created_at', 'updated_at')
    list_per_page = 20

    fieldsets = (
        (None, {
            'fields': ('name', 'description', 'is_active')
        }),
        (_('Timestamps'), {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )

@admin.register(Expense)
class ExpenseAdmin(admin.ModelAdmin):
    list_display = ('reference', 'description', 'category', 'amount', 'expense_date', 'status', 'created_by', 'created_at')
    list_filter = ('status', 'category', 'payment_method', 'expense_date')
    search_fields = ('reference', 'description', 'notes')
    readonly_fields = ('created_at', 'updated_at', 'created_by')
    list_per_page = 20
    date_hierarchy = 'expense_date'
    
    fieldsets = (
        (_('Basic Information'), {
            'fields': ('reference', 'description', 'category', 'amount', 'expense_date')
        }),
        (_('Payment Details'), {
            'fields': ('payment_method', 'status')
        }),
        (_('Additional Information'), {
            'fields': ('notes', 'created_by'),
            'classes': ('collapse',)
        }),
        (_('Timestamps'), {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )
    
    def save_model(self, request, obj, form, change):
        if not obj.pk:  # Only set created_by if this is a new object
            obj.created_by = request.user
        super().save_model(request, obj, form, change)
